package com.aliyun.core.utils;

import com.aliyun.core.exception.AliyunException;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.reflect.TypeToken;
import org.dom4j.DocumentException;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.*;

public class ParseUtil {

    /**
     * Parse it by JSON format
     *
     * @return the parsed result
     */
    public static Object parseJSON(String json) {
        Gson gson = new GsonBuilder()
                .registerTypeAdapter(new TypeToken<Map<String, Object>>() {
                }.getType(), new MapTypeAdapter()).create();

        JsonElement jsonElement = gson.fromJson(json, JsonElement.class);
        return jsonElement.isJsonArray() ? gson.fromJson(json, List.class) :
                gson.fromJson(json, new TypeToken<Map<String, Object>>() {
                }.getType());
    }

    /**
     * Read data from a readable stream, and compose it to a bytes
     *
     * @param stream the readable stream
     * @return the bytes result
     */
    public static byte[] readAsBytes(InputStream stream) {
        try {
            if (null == stream) {
                return new byte[]{};
            } else {
                ByteArrayOutputStream os = new ByteArrayOutputStream();
                byte[] buff = new byte[1024];
                while (true) {
                    int read = stream.read(buff);
                    if (read == -1) {
                        return os.toByteArray();
                    }
                    os.write(buff, 0, read);
                }
            }
        } catch (Exception e) {
            throw new AliyunException(e.getMessage(), e);
        } finally {
            if (null != stream) {
                try {
                    stream.close();
                } catch (IOException e) {
                    throw new AliyunException(e.getMessage(), e);
                }
            }
        }
    }

    /**
     * Read data from a readable stream, and compose it to a string
     *
     * @param stream the readable stream
     * @return the string result
     */
    public static String readAsString(InputStream stream) {
        try {
            return new String(readAsBytes(stream), "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new AliyunException(e.getMessage(), e);
        }
    }

    /**
     * Read data from a readable stream, and parse it by JSON format
     *
     * @param str the readable stream
     * @return the parsed result
     */
    public static Object readAsJSON(String str) {
//        String body = readAsString(stream);
        try {
            return parseJSON(str);
        } catch (Exception exception) {
            throw new AliyunException("Error: convert to JSON, response is:\n" + str);
        }
    }

    /**
     * If not set the real, use default value
     *
     * @return the return string
     */
    public static String toJSONString(Object object) {
        if (object instanceof String) {
            return (String) object;
        }
        return new Gson().toJson(object);
    }

    public static Map<String, Object> parseXml(String str) throws DocumentException {
        return XmlUtil.deserializeXml(str);
    }

    public static Map<String, Object> readAsXML(String str) {
        try {
            return parseXml(str);
        } catch (Exception exception) {
            throw new AliyunException("Error: XML convert to MAP, response is:\n" + str);
        }
    }

    public static String toXmlString(Map<String, Object> xmlMap, String rootName) {
        return XmlUtil.serializeXml(xmlMap, rootName);
    }
}
