package darabonba.core;

import darabonba.core.utils.CommonUtil;

import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import static darabonba.core.utils.ModelUtil.shrinkSpecifiedStyle;

public class RequestModel extends TeaModel {
    private final Map<String, Object> queryParameters;
    private final Map<String, Object> bodyParameters;
    private final Map<String, String> hostParameters;
    private final Map<String, String> pathParameters;
    private final Map<String, Object> headerParameters;

    private RequestConfiguration requestConfiguration;

    protected RequestModel(BuilderImpl<?, ?> builder) {
        this.queryParameters = builder.queryParameters;
        this.bodyParameters = builder.bodyParameters != null && !builder.bodyParameters.isEmpty() ? builder.bodyParameters : null;
        this.hostParameters = builder.hostParameters;
        this.pathParameters = builder.pathParameters;
        this.headerParameters = builder.headerParameters;
        this.requestConfiguration = builder.requestConfiguration;
    }

    public RequestConfiguration getRequestConfiguration() {
        return this.requestConfiguration;
    }

    public Map<String, Object> getQueryParameters() {
        return this.queryParameters;
    }

    public Map<String, Object> getBodyParameters() {
        return this.bodyParameters;
    }

    public Map<String, String> getHostParameters() {
        return this.hostParameters;
    }

    public Map<String, String> getPathParameters() {
        return this.pathParameters;
    }

    public Map<String, Object> getHeaderParameters() {
        return this.headerParameters;
    }

    public interface Builder<ProviderT extends RequestModel, BuilderT extends Builder> {
        BuilderT requestConfiguration(RequestConfiguration requestConfiguration);

        ProviderT build();
    }

    protected abstract static class BuilderImpl<ProviderT extends RequestModel, BuilderT extends Builder>
            implements Builder<ProviderT, BuilderT> {
        private Map<String, Object> queryParameters = new HashMap();
        private Map<String, Object> bodyParameters = new HashMap();
        private Map<String, String> hostParameters = new HashMap();
        private Map<String, String> pathParameters = new HashMap();
        private Map<String, Object> headerParameters = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        private RequestConfiguration requestConfiguration;

        protected BuilderImpl() {
        }

        protected BuilderImpl(RequestModel requestModel) {
            this.queryParameters = requestModel.queryParameters;
            this.bodyParameters = requestModel.bodyParameters;
            this.hostParameters = requestModel.hostParameters;
            this.pathParameters = requestModel.pathParameters;
            this.headerParameters = requestModel.headerParameters;
            this.requestConfiguration = requestModel.requestConfiguration;
        }

        @Override
        public BuilderT requestConfiguration(RequestConfiguration requestConfiguration) {
            this.requestConfiguration = requestConfiguration;
            return (BuilderT) this;
        }

        protected void putQueryParameter(String name, Object value) {
            if (!CommonUtil.isUnset(name)) {
                this.queryParameters.put(name, value);
            }
        }

        protected void putBodyParameter(String name, Object value) {
            if (!CommonUtil.isUnset(name)) {
                this.bodyParameters.put(name, value);
            }
        }

        protected void putHostParameter(String name, Object value) {
            if (!CommonUtil.isUnset(name)) {
                this.hostParameters.put(name, String.valueOf(value));
            }
        }

        protected void putPathParameter(String name, Object value) {
            if (!CommonUtil.isUnset(name)) {
                this.pathParameters.put(name, String.valueOf(value));
            }
        }

        protected void putHeaderParameter(String name, Object value) {
            if (!CommonUtil.isUnset(name)) {
                this.headerParameters.put(name, value);
            }
        }

        protected String shrink(Object o, String prefix, String style) {
            return shrinkSpecifiedStyle(o, prefix, style);
        }
    }
}