package darabonba.core.async;

import com.aliyun.core.utils.BinaryUtils;
import darabonba.core.TeaRequestBody;
import darabonba.core.internal.async.FileAsyncRequestBody;
import darabonba.core.internal.async.ByteArrayAsyncRequestBody;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;

import java.io.File;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Optional;

public interface AsyncRequestBody extends Publisher<ByteBuffer>, TeaRequestBody {

    static AsyncRequestBody fromPublisher(Publisher<ByteBuffer> publisher) {
        return new AsyncRequestBody() {

            @Override
            public Optional<Long> contentLength() {
                return Optional.empty();
            }

            @Override
            public Optional<String> contentType() {
                return Optional.empty();
            }

            @Override
            public void subscribe(Subscriber<? super ByteBuffer> s) {
                publisher.subscribe(s);
            }
        };
    }

    static AsyncRequestBody fromFile(Path path) {
        return new FileAsyncRequestBody(path);
    }

    static AsyncRequestBody fromFile(File file) {
        return new FileAsyncRequestBody(file.toPath());
    }

    static AsyncRequestBody fromString(String string, Charset cs) {
        return new ByteArrayAsyncRequestBody(string.getBytes(cs));
    }

    static AsyncRequestBody fromString(String string) {
        return fromString(string, StandardCharsets.UTF_8);
    }

    static AsyncRequestBody fromBytes(byte[] bytes) {
        return new ByteArrayAsyncRequestBody(bytes);
    }

    static AsyncRequestBody fromByteBuffer(ByteBuffer byteBuffer) {
        return new ByteArrayAsyncRequestBody(BinaryUtils.copyAllBytesFrom(byteBuffer));
    }

    static AsyncRequestBody empty() {
        return fromBytes(new byte[0]);
    }
}
