package darabonba.core.policy.retry;

import com.aliyun.core.logging.ClientLogger;
import darabonba.core.policy.retry.backoff.BackoffStrategy;
import darabonba.core.policy.retry.conditions.AndRetryCondition;
import darabonba.core.policy.retry.conditions.MaxRetriesCondition;
import darabonba.core.policy.retry.conditions.RetryCondition;

public final class RetryPolicy {
    private final ClientLogger logger = new ClientLogger(RetryPolicy.class);
    private final BackoffStrategy backoffStrategy;
    private final Integer numRetries;
    private final RetryCondition retryCondition;
    private final RetryCondition aggregateRetryCondition;

    private RetryPolicy(BuilderImpl builder) {
        this.backoffStrategy = builder.backoffStrategy;
        this.numRetries = builder.numRetries;
        this.retryCondition = builder.retryCondition;
        this.aggregateRetryCondition = generateAggregateRetryCondition();
    }

    public static RetryPolicy defaultRetryPolicy() {
        return builder().build();
    }

    public static RetryPolicy none() {
        return builder()
                .numRetries(0)
                .backoffStrategy(BackoffStrategy.none())
                .retryCondition(RetryCondition.none())
                .build();
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public RetryCondition aggregateRetryCondition() {
        return aggregateRetryCondition;
    }

    public RetryCondition retryCondition() {
        return retryCondition;
    }

    public BackoffStrategy backoffStrategy() {
        return backoffStrategy;
    }

    public Integer numRetries() {
        return numRetries;
    }

    private RetryCondition generateAggregateRetryCondition() {
        RetryCondition aggregate = AndRetryCondition.create(MaxRetriesCondition.create(numRetries),
                retryCondition);
        return aggregate;
    }

    public Builder toBuilder() {
        return builder().numRetries(numRetries)
                .retryCondition(retryCondition)
                .backoffStrategy(backoffStrategy);
    }

    public interface Builder {

        Builder backoffStrategy(BackoffStrategy backoffStrategy);

        Builder retryCondition(RetryCondition retryCondition);

        Builder numRetries(Integer numRetries);

        RetryPolicy build();
    }

    /**
     * Builder for a {@link RetryPolicy}.
     */
    public static final class BuilderImpl implements Builder {
        private Integer numRetries;
        private BackoffStrategy backoffStrategy;
        private BackoffStrategy throttlingBackoffStrategy;
        private RetryCondition retryCondition;

        private BuilderImpl() {
            this.numRetries = RetryUtil.DEFAULT_MAX_RETRIES;
            this.backoffStrategy = BackoffStrategy.defaultStrategy();
            this.throttlingBackoffStrategy = BackoffStrategy.defaultThrottlingStrategy();
            this.retryCondition = RetryCondition.defaultRetryCondition();
        }

        @Override
        public Builder numRetries(Integer numRetries) {
            this.numRetries = numRetries;
            return this;
        }

        @Override
        public Builder backoffStrategy(BackoffStrategy backoffStrategy) {
            this.backoffStrategy = backoffStrategy;
            return this;
        }

        @Override
        public Builder retryCondition(RetryCondition retryCondition) {
            this.retryCondition = retryCondition;
            return this;
        }

        @Override
        public RetryPolicy build() {
            return new RetryPolicy(this);
        }
    }
}
