package com.aliyun.datahub.client.common;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public abstract class ExpireCache<K, V> {
    private final int expireTime;
    private final int cleanTime = 24*3600;
    private final List<CacheBucket<V>> bucketList;

    public ExpireCache(int bucketSize, int expireTime) {
        this.bucketList = new ArrayList<>(bucketSize);
        this.expireTime = expireTime;
    }

    public abstract V fetchItem(K key);

    public V getItem(K key) {
        int index = key.hashCode() % bucketList.size();
        CacheBucket<V> bucket = bucketList.get(index);
        if (bucket == null) {
            bucket = new CacheBucket<>(this);
            bucketList.add(bucket);
        }
        return bucket.getItem(key);
    }

    public int getExpireTime() {
        return expireTime;
    }

    public int getCleanTime() {
        return cleanTime;
    }

    private class CacheBucket<V> {
        private final ExpireCache<K, V> cache;
        private long lastCleanTime;
        private final Map<K, CacheItem<K, V>> itemMap;

        public CacheBucket(ExpireCache<K, V> cache) {
            this.cache = cache;
            this.itemMap = new HashMap<>();
            this.lastCleanTime = System.currentTimeMillis()/1000;
        }

        public synchronized V getItem(K key) {
            checkClean();
            CacheItem<K, V> item = itemMap.get(key);
            if (item == null) {
                item = new CacheItem<>(cache, key, cache.getExpireTime());
                itemMap.put(key, item);
            }
            return item.get();
        }

        private void checkClean() {
            long currTime = System.currentTimeMillis()/1000;
            if (currTime - lastCleanTime > this.cache.getCleanTime()) {
                doClean(currTime);
                this.lastCleanTime = currTime;
            }
        }

        private void doClean(long curr) {
            itemMap.entrySet().removeIf(entry -> curr - entry.getValue().getLastUpdateTime() > this.cache.getCleanTime());
        }
    }

    private static class CacheItem<K, V> {
        private final long lastUpdateTime;
        private final ExpireCache<K, V> cache;
        private final Supplier<V> supplier;

        public CacheItem(ExpireCache<K, V> cache, K key, int expireTime) {
            this.lastUpdateTime = System.currentTimeMillis()/1000;
            this.cache = cache;
            this.supplier = Suppliers.memoizeWithExpiration(
                    () -> this.cache.fetchItem(key),
                    expireTime,
                    TimeUnit.SECONDS);
        }

        public V get() {
            return this.supplier.get();
        }

        public long getLastUpdateTime() {
            return lastUpdateTime;
        }
    }
}
