package com.aliyun.datahub.client.http.converter.batch;

import com.aliyun.datahub.client.exception.DatahubClientException;
import com.aliyun.datahub.client.model.CompressType;
import com.aliyun.datahub.client.util.CrcUtils;
import net.jpountz.lz4.LZ4Compressor;
import net.jpountz.lz4.LZ4Factory;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;

import static java.util.zip.Deflater.BEST_SPEED;

public class BatchBinaryRecord {
    private static final LZ4Compressor LZ4_COMPRESSOR = LZ4Factory.fastestInstance().fastCompressor();
    public final static int BATCH_HEADER_SIZE = 26;
    private final static byte[] MAGIC_NUMBER = new byte[] { 'D', 'H', 'U', 'B' };

    private List<BinaryRecord> records = new ArrayList<>();

    public void addRecord(BinaryRecord record) {
        records.add(record);
    }

    public List<BinaryRecord> getRecords() {
        return records;
    }

    public byte[] serialize(CompressType type) {
        try {
            int calSize = BATCH_HEADER_SIZE;
            for (BinaryRecord record : records) {
                calSize += record.getRecordSize();
            }
            BatchOutput output = new BatchOutput(calSize, records.size());
            for (BinaryRecord record : records) {
                record.serialize(output);
            }
            output.compressIfNeed(type);
            return output.toByteArray();
        } catch (Exception e) {
            throw new DatahubClientException("Serialize batch record fail. error:" + e.getMessage());
        }
    }

    public static class BatchHeader {
        private static final ThreadLocal<ByteBuffer> BYTE_BUFFER = ThreadLocal.withInitial(() -> ByteBuffer.allocate(BATCH_HEADER_SIZE).order(ByteOrder.LITTLE_ENDIAN));
        private byte[] magic = MAGIC_NUMBER;
        private int version;
        private int length;
        private int rawSize;
        private int crc32;
        private short attributes;
        private int recordCount;

        public byte[] getMagic() {
            return magic;
        }

        public void setMagic(byte[] magic) {
            this.magic = magic;
        }

        public int getVersion() {
            return version;
        }

        public void setVersion(int version) {
            this.version = version;
        }

        public int getLength() {
            return length;
        }

        public void setLength(int length) {
            this.length = length;
        }

        public int getRawSize() {
            return rawSize;
        }

        public void setRawSize(int rawSize) {
            this.rawSize = rawSize;
        }

        public int getCrc32() {
            return crc32;
        }

        public void setCrc32(int crc32) {
            this.crc32 = crc32;
        }

        public short getAttributes() {
            return attributes;
        }

        public void setAttributes(short attributes) {
            this.attributes = attributes;
        }

        public int getRecordCount() {
            return recordCount;
        }

        public void setRecordCount(int recordCount) {
            this.recordCount = recordCount;
        }

        public static BatchHeader parseFrom(byte[] bytes) {
            ByteBuffer byteBuffer = BYTE_BUFFER.get();
            ByteArrayInputStream input = new ByteArrayInputStream(bytes);
            byte[] buffer = new byte[BATCH_HEADER_SIZE];
            int len = input.read(buffer, 0, BATCH_HEADER_SIZE);
            if (len < BATCH_HEADER_SIZE) {
                throw new DatahubClientException("read batch header fail");
            }
            byteBuffer.clear();
            byteBuffer.put(buffer);
            byteBuffer.flip();
            BatchHeader header = new BatchHeader() {{
                setMagic(BatchUtil.parseInt(byteBuffer.getInt()));
                setVersion(byteBuffer.getInt());
                setLength(byteBuffer.getInt());
                setRawSize(byteBuffer.getInt());
                setCrc32(byteBuffer.getInt());
                setAttributes(byteBuffer.getShort());
                setRecordCount(byteBuffer.getInt());
            }};

            if (!Arrays.equals(header.getMagic(), MAGIC_NUMBER)) {
                throw new DatahubClientException("Check magic number fail");
            }
            if (bytes.length != header.getLength()) {
                throw new DatahubClientException("Check payload length fail");
            }

            if (header.getCrc32() != 0) {
                int computeCrc = CrcUtils.getCrc32(bytes, BATCH_HEADER_SIZE, header.getLength() - BATCH_HEADER_SIZE);
                if (header.getCrc32() != computeCrc) {
                    throw new DatahubClientException("Check crc fail. expect:" + header.getCrc32() + ", real:" + computeCrc);
                }
            }

            return header;
        }

        public static byte[] serialize(BatchHeader header) {
            ByteBuffer byteBuffer = BYTE_BUFFER.get();
            byteBuffer.clear();
            byteBuffer.put(header.getMagic());
            byteBuffer.putInt(header.getVersion());
            byteBuffer.putInt(header.getLength());
            byteBuffer.putInt(header.getRawSize());
            byteBuffer.putInt(header.getCrc32());
            byteBuffer.putShort(header.getAttributes());
            byteBuffer.putInt(header.getRecordCount());
            return byteBuffer.array();
        }
    }

    private static class BatchOutput extends ByteArrayOutputStream {
        private final BatchHeader header = new BatchHeader();
        private final int recordCount;

        public BatchOutput(int calSize, int recordCount) {
            super(BATCH_HEADER_SIZE + calSize);
            initHeader();
            this.recordCount = recordCount;
        }

        private void initHeader() {
            byte[] buffer = BatchHeader.serialize(header);
            write(buffer, 0, buffer.length);
        }

        public void compressIfNeed(CompressType type) {
            if (type == null) {
                type = CompressType.NONE;
            }
            int totalSize = getRawSize();
            int rawSize = totalSize - BATCH_HEADER_SIZE;
            header.setLength(totalSize);
            header.setRawSize(rawSize);
            header.setAttributes((short)(0x08 | type.getValue()));

            if (type != CompressType.NONE) {
                byte[] compressData = null;
                try {
                    if (type == CompressType.DEFLATE) {
                        ByteArrayOutputStream out = new ByteArrayOutputStream();
                        DeflaterOutputStream dos = new DeflaterOutputStream(out, new Deflater(BEST_SPEED));
                        dos.write(buf, BATCH_HEADER_SIZE, rawSize);
                        dos.close();
                        compressData = out.toByteArray();
                    } else {
                        compressData = LZ4_COMPRESSOR.compress(buf, BATCH_HEADER_SIZE, rawSize);
                    }

                    if (compressData.length > rawSize) {
                        header.setAttributes((short)(0x08 | CompressType.NONE.getValue()));
                    } else {
                        System.arraycopy(compressData, 0, buf, BATCH_HEADER_SIZE, compressData.length);
                        count = BATCH_HEADER_SIZE + compressData.length;
                        header.setLength(count);
                    }
                } catch (Exception e) {
                    throw new DatahubClientException("Compress data fail. error:" + e.getMessage());
                }
            }

        }

        private int getRawSize() {
            return this.size();
        }

        @Override
        public synchronized byte[] toByteArray() {
            header.setMagic(MAGIC_NUMBER);
            header.setVersion(0);
            header.setRecordCount(recordCount);
            int crc32 = CrcUtils.getCrc32(buf, BATCH_HEADER_SIZE, count - BATCH_HEADER_SIZE);
            header.setCrc32(crc32);

            byte[] buffer = BatchHeader.serialize(header);
            System.arraycopy(buffer, 0, buf, 0, BATCH_HEADER_SIZE);
            return super.toByteArray();
        }
    }
}
