package com.aliyun.datahub.client.http.converter.batch;

import com.aliyun.datahub.client.impl.schemaregistry.SchemaRegistryClient;
import com.aliyun.datahub.client.model.*;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

public class BatchSerializer {
    private final String projectName;
    private final String topicName;
    private final CompressType compressType;
    private final SchemaRegistryClient schemaRegistry;

    public BatchSerializer(String projectName, String topicName,
                           CompressType compressType,
                           SchemaRegistryClient schemaRegistry) {
        this.projectName = projectName;
        this.topicName = topicName;
        this.compressType = compressType;
        this.schemaRegistry = schemaRegistry;
    }

    public byte[] serialize(List<RecordEntry> recordList) {
        BatchBinaryRecord batch = new BatchBinaryRecord();

        ForkJoinPool pool = BatchSerializePool.getWritePool();
        List<BinaryRecord> binaryRecords;
        if (pool == null) {
            binaryRecords = recordList.stream().map(this::convertToBinaryRecord).collect(Collectors.toList());
        } else {
            binaryRecords = pool.submit(() -> recordList.parallelStream().map(this::convertToBinaryRecord).collect(Collectors.toList())).join();
        }

        for (BinaryRecord record : binaryRecords) {
            batch.addRecord(record);
        }
        return batch.serialize(compressType);
    }

    private BinaryRecord convertToBinaryRecord(RecordEntry entry) {
        RecordData recordData = entry.getRecordData();
        BinaryRecord record;
        if (recordData instanceof TupleRecordData) {
            record = convertTupleToBinaryRecord((TupleRecordData)(entry.getRecordData()));
        } else {
            record = convertBlobToBinaryRecord((BlobRecordData)(entry.getRecordData()));
        }

        if (entry.getAttributes() != null) {
            for (Map.Entry<String, String> item : entry.getAttributes().entrySet()) {
                record.addAttribute(item.getKey(), item.getValue());
            }
        }

        return record;
    }

    private BinaryRecord convertTupleToBinaryRecord(TupleRecordData data) {
        RecordSchema schema = data.getRecordSchema();
        int versionId = 0;
        if (this.schemaRegistry != null) {
            versionId = this.schemaRegistry.getVersionId(projectName, topicName, schema);
        }
        BinaryRecord record = new BinaryRecord(schema, versionId);
        for (int i = 0; i < schema.getFields().size(); ++i) {
            Object value = data.getField(i);
            if (value != null) {
                record.setField(i, value);
            }
        }
        return record;
    }

    private static BinaryRecord convertBlobToBinaryRecord(BlobRecordData data) {
        BinaryRecord record = new BinaryRecord(null, -1);
        record.setField(0, data.getData());
        return record;
    }
}
