package com.aliyun.datahub.client.impl;

import com.aliyun.datahub.client.auth.Account;
import com.aliyun.datahub.client.common.DatahubConfig;
import com.aliyun.datahub.client.exception.DatahubClientException;
import com.aliyun.datahub.client.exception.InvalidParameterException;
import com.aliyun.datahub.client.http.HttpConfig;
import com.aliyun.datahub.client.http.converter.batch.BatchDeserializer;
import com.aliyun.datahub.client.http.converter.batch.BatchSerializer;
import com.aliyun.datahub.client.impl.request.protobuf.GetBatchRecordsRequestPB;
import com.aliyun.datahub.client.impl.request.protobuf.PutBatchRecordsRequestPB;
import com.aliyun.datahub.client.model.*;
import com.aliyun.datahub.client.model.protobuf.GetBatchRecordsResultPB;
import com.aliyun.datahub.client.util.FormatUtils;
import com.codahale.metrics.Timer;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

public class DatahubClientBatchImpl extends DatahubClientJsonImpl {

    public DatahubClientBatchImpl(String endpoint, Account account, DatahubConfig datahubConfig, HttpConfig httpConfig, String userAgent) {
        super(endpoint, account, datahubConfig, httpConfig, userAgent);
    }

    @Override
    public PutRecordsResult putRecords(String projectName, String topicName, List<RecordEntry> records) {
        throw new DatahubClientException("This method is not supported for batch client, please use putRecordsByShard");
    }

    @Override
    public PutRecordsByShardResult putRecordsByShard(String projectName, String topicName, String shardId, List<RecordEntry> records) {
        if (!FormatUtils.checkProjectName(projectName)) {
            throw new InvalidParameterException("ProjectName format is invalid");
        }

        if (!FormatUtils.checkTopicName(topicName)) {
            throw new InvalidParameterException("TopicName format is invalid");
        }

        if (!FormatUtils.checkShardId(shardId)) {
            throw new InvalidParameterException("ShardId format is invalid");
        }

        if (records == null || records.isEmpty()) {
            throw new InvalidParameterException("Records is null or empty");
        }

        BatchSerializer serializer = new BatchSerializer(projectName, topicName, httpConfig.getCompressType(), getSchemaRegistry());
        PutBatchRecordsRequestPB request = new PutBatchRecordsRequestPB()
                .setSerializer(serializer)
                .setRecords(records);
        final Timer.Context context = PUT_LATENCY_TIMER == null ? null : PUT_LATENCY_TIMER.time();
        try {
            PutRecordsByShardResult result = callWrapper(getService().putBatchRecordsByShard(projectName,
                    topicName, shardId, request));
            if (result != null) {
                if (PUT_QPS_METER != null) {
                    PUT_QPS_METER.mark(1);
                }

                if (PUT_RPS_METER != null) {
                    PUT_RPS_METER.mark(records.size());
                }
            }
            return result;
        } finally {
            if (context != null) {
                context.stop();
            }
        }
    }

    @Override
    public GetRecordsResult getRecords(String projectName, String topicName, String shardId, String cursor, int limit) {
        return getRecords(projectName, topicName, shardId, null, cursor, limit);
    }

    @Override
    public GetRecordsResult getRecords(String projectName, String topicName, String shardId, RecordSchema schema, String cursor, int limit) {
        if (!FormatUtils.checkProjectName(projectName)) {
            throw new InvalidParameterException("ProjectName format is invalid");
        }

        if (!FormatUtils.checkTopicName(topicName)) {
            throw new InvalidParameterException("TopicName format is invalid");
        }

        if (StringUtils.isEmpty(cursor)) {
            throw new InvalidParameterException("Cursor format is invalid");
        }

        limit = Math.max(MIN_FETCH_SIZE, limit);
        limit = Math.min(MAX_FETCH_SIZE, limit);

        final GetBatchRecordsRequestPB request = new GetBatchRecordsRequestPB();
        request.setCursor(cursor).setLimit(limit);

        final Timer.Context context = GET_LATENCY_TIMER == null ? null : GET_LATENCY_TIMER.time();
        try {
            GetBatchRecordsResultPB result = callWrapper(getService().getBatchRecords(projectName,
                    topicName, shardId, request));

            if (result != null) {
                result.setDeserializer(new BatchDeserializer(projectName, topicName, schema, getSchemaRegistry()));
                result.internalSetShardId(shardId);
                if (GET_QPS_METER != null) {
                    GET_QPS_METER.mark(1);
                }
                if (GET_RPS_METER != null) {
                    GET_RPS_METER.mark(result.getRecordCount());
                }
            }
            return result;
        } finally {
            if (context != null) {
                context.stop();
            }
        }
    }
}
