package com.aliyun.datahub.client.impl.interceptor;

import com.aliyun.datahub.client.common.DatahubConstant;
import com.aliyun.datahub.client.common.HttpHeaders;
import com.aliyun.datahub.client.exception.DatahubClientException;
import com.aliyun.datahub.client.http.interceptor.HttpInterceptor;
import com.aliyun.datahub.client.util.JsonUtils;
import com.fasterxml.jackson.annotation.JsonProperty;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicInteger;

import static com.aliyun.datahub.client.http.common.Constants.CONTENT_JSON;

public class DatahubResponseInterceptor extends HttpInterceptor {
    private final static Logger LOGGER = LoggerFactory.getLogger(DatahubResponseInterceptor.class);
    private static String ipAddress;
    private static String processId;
    private static String clientId;
    private String userAgent;
    private String traceIdPrefix;
    private AtomicInteger id = new AtomicInteger();

    static {
        ipAddress = getIpAddress();
        processId = getProcessId();
        clientId = getClientId();
    }

    private static String getIpAddress() {
        String ip = null;
        Enumeration<NetworkInterface> interfaces = null;
        try {
            interfaces = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException e) {
            throw new RuntimeException(e);
        }

        while (interfaces.hasMoreElements()) {
            NetworkInterface element = interfaces.nextElement();
            Enumeration<InetAddress> addresses = element.getInetAddresses();
            while (addresses.hasMoreElements()) {
                InetAddress address = addresses.nextElement();
                if (address.isLoopbackAddress()) {
                    continue;
                }
                if (address instanceof Inet4Address) {
                    ip = address.getHostAddress();
                }
            }
        }
        return ip;
    }

    private static String getProcessId() {
        String processName = java.lang.management.ManagementFactory
                .getRuntimeMXBean().getName();
        return processName.substring(0, processName.indexOf('@'));
    }

    private static String getClientId() {
        // version
        String path = "META-INF/maven/com.aliyun.datahub/aliyun-sdk-datahub/pom.properties";
        Properties p = new Properties();
        try {
            p.load(DatahubAuthInterceptor.class.getClassLoader().getResourceAsStream(path));
        } catch (Throwable ignored) {
        }

        String versionNumber = p.getProperty("version", "Unknown");
        return String.format(Locale.ENGLISH, "%s@%s-(Java-%s)", ipAddress, processId, versionNumber);
    }
    
    public DatahubResponseInterceptor(String userAgent) {
        if (clientId != null) {
            this.userAgent = (userAgent == null ? clientId : clientId + "-" + userAgent);
        } else if (userAgent != null) {
            this.userAgent = userAgent;
        }

        traceIdPrefix = this.userAgent == null ? "00000000" : String.format("%08x", this.userAgent.hashCode());
    }


    String genClientTraceId() {
        return traceIdPrefix + String.format("%016x", ((System.currentTimeMillis() << 16) | (id.getAndIncrement() & 0xffff)));
    }
    
    @Override
    public Response intercept(Chain chain) throws IOException {
        String clientTraceId = genClientTraceId();
        Request request = addHeader(chain, clientTraceId);
        try {
            Response response = chain.proceed(request);
            String requestId = response.header(DatahubConstant.X_DATAHUB_REQUEST_ID);
            if (requestId == null) {
                LOGGER.warn("requestId is null");
            }
            int status = response.code();
            if (status >= 400) {
                DatahubResponseInterceptor.DatahubResponseError error = new DatahubResponseInterceptor.DatahubResponseError();
                String contentType = response.header(HttpHeaders.CONTENT_TYPE.toLowerCase());
                ResponseBody body = response.body();
                if (body != null) {
                    if (CONTENT_JSON.equalsIgnoreCase(contentType)) {
                        String bodyStr = body.string();
                        error = JsonUtils.fromJson(bodyStr, DatahubResponseInterceptor.DatahubResponseError.class);
                    } else {
                        error = new DatahubResponseInterceptor.DatahubResponseError();
                        error.setMessage(body.string());
                    }
                }
                throw new DatahubClientException(status,
                        requestId,
                        clientTraceId,
                        error == null ? "" : error.getCode(),
                        error == null ? "" : error.getMessage());
            }
            return response;
        } catch (IOException e) {
            throw new DatahubClientException(DatahubConstant.DEFAULT_CLIENT_ERROR_STATUS,
                    null, clientTraceId,
                    null, e.getMessage() + ", url: " + request.url(),
                    e);
        }
    }

    private Request addHeader(Chain chain, String clientTraceId) {
        Request.Builder reqBuilder = chain.request().newBuilder();
        if (ipAddress != null) {
            reqBuilder.addHeader(DatahubConstant.X_DATAHUB_SOURCE_IP, ipAddress);
        }

        if (this.userAgent != null) {
            reqBuilder.addHeader(HttpHeaders.USER_AGENT, userAgent + "-" + clientTraceId);
        }
        return reqBuilder.build();
    }

    private static class DatahubResponseError {
        @JsonProperty("RequestId")
        private String requestId;

        @JsonProperty("ErrorCode")
        private String code;

        @JsonProperty("ErrorMessage")
        private String message;

        public String getRequestId() {
            return requestId;
        }

        public void setRequestId(String requestId) {
            this.requestId = requestId;
        }

        public String getCode() {
            return code;
        }

        public void setCode(String code) {
            this.code = code;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }
}
