package com.aliyun.datahub.client.impl.serializer;

import com.aliyun.datahub.client.model.*;
import com.aliyun.datahub.client.util.JsonUtils;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;

import java.io.IOException;
import java.util.*;

public class GetConnectorResultDeserializer extends JsonDeserializer<GetConnectorResult> {
    @Override
    public GetConnectorResult deserialize(JsonParser jsonParser, DeserializationContext deserializationContext) throws IOException, JsonProcessingException {
        GetConnectorResult result = new GetConnectorResult();
        JsonNode tree = jsonParser.getCodec().readTree(jsonParser);
        JsonNode node = tree.get("ConnectorId");
        if (node != null && !node.isNull()) {
            result.setConnectorId(node.asText());
        }
        node = tree.get("ClusterAddress");
        if (node != null && !node.isNull()) {
            result.setClusterAddr(node.asText());
        }
        node = tree.get("Creator");
        if (node != null && !node.isNull()) {
            result.setCreator(node.asText());
        }
        node = tree.get("CreateTime");
        if (node != null && node.isNumber()) {
            result.setCreateTime(node.asLong());
        }
        node = tree.get("LastModifyTime");
        if (node != null && node.isNumber()) {
            result.setLastModifyTime(node.asLong());
        }
        node = tree.get("ColumnFields");
        if (node != null && node.isArray()) {
            List<String> columnFields = new ArrayList<>();
            for (JsonNode fNode : node) {
                columnFields.add(fNode.asText());
            }
            result.setColumnFields(columnFields);
        }
        node = tree.get("ColumnNameMap");
        if (node != null && node.isObject()) {
            Map<String, String> columnNameMap = new HashMap<>();
            Iterator<Map.Entry<String, JsonNode>> iter = node.fields();
            while (iter.hasNext()) {
                Map.Entry<String, JsonNode> entry = iter.next();
                if (entry.getValue() != null && entry.getValue().isTextual()) {
                    columnNameMap.put(entry.getKey(), entry.getValue().asText());
                }
            }
            result.setColumnNameMap(columnNameMap);
        }
        node = tree.get("Type");
        if (node != null && !node.isNull()) {
            result.setType(ConnectorType.valueOf(node.asText().toUpperCase()));
        }
        node = tree.get("State");
        if (node != null && !node.isNull()) {
            result.setState(ConnectorState.fromString(node.asText().toUpperCase()));
        }
        node = tree.get("ShardContexts");
        if (node != null && node.isArray()) {
            List<ShardContext> shardContexts = new ArrayList<>();
            for (JsonNode fNode : node) {
                ShardContext context = JsonUtils.fromJsonNode(fNode, ShardContext.class);
                if (context == null) {
                    throw new IOException("ShardContext format is invalid");
                }
                shardContexts.add(context);
            }
            result.setShardContexts(shardContexts);
        }
        node = tree.get("ExtraInfo");
        if (node != null && node.isObject()) {
            Map<String, String> extraConfig = new HashMap<>();
            Iterator<Map.Entry<String, JsonNode>> iter = node.fields();
            while (iter.hasNext()) {
                Map.Entry<String, JsonNode> entry = iter.next();
                if (entry.getValue() != null && entry.getValue().isTextual()) {
                    extraConfig.put(entry.getKey(), entry.getValue().asText());
                }
            }
            if (!extraConfig.isEmpty()) {
                result.setExtraConfig(extraConfig);
                result.setSubId(extraConfig.get("SubscriptionId"));
            }
        }

        result.setConfig(deserializeConnectorConfig(result.getType(), tree));
        return result;
    }

    private ConnectorConfig deserializeConnectorConfig(ConnectorType type, JsonNode tree) {
        JsonNode node = tree.get("Config");
        ConnectorConfig config = null;
        if (node != null && node.isObject()) {
            switch (type) {
                case SINK_ODPS:
                    config = deserializeSinkOdpsConfig(node);
                    break;
                case SINK_ADS:
                case SINK_MYSQL:
                    config = deserializeSinkDatabaseConfig(node);
                    break;
                case SINK_DATAHUB:
                    config = deserializeSinkDatahubConfig(node);
                    break;
                case SINK_HOLOGRES:
                    config = deserializeSinkHologresConfig(node);
                    break;
                case SINK_ES:
                    config = deserializeSinkEsConfig(node);
                    break;
                case SINK_FC:
                    config = deserializeSinkFcConfig(node);
                    break;
                case SINK_OTS:
                    config = deserializeSinkOtsConfig(node);
                    break;
                case SINK_OSS:
                    config = deserializeSinkOssConfig(node);
                    break;
                case SOURCE_DTS:
                    config = deserializeSourceDtsConfig(node);
                    break;
            }
            deserializeCommonConfig(config, node);
        }
        return config;
    }

    private BinaryParserConfig parseBinaryParserConfig(JsonNode tree) {
        JsonNode node = tree.get("BinaryParserType");
        if (node == null || node.isNull()) {
            return null;
        }

        BinaryParserType type = BinaryParserType.fromString(node.asText());
        if (type == null) {
            return null;
        }

        JsonNode propertiesNode = tree.get("Properties");
        BinaryParserConfig result = null;
        switch (type) {
            case DELIMITER_PARSER:
                DelimiterParserConfig config = new DelimiterParserConfig();
                if (propertiesNode != null && propertiesNode.isObject()) {
                    node = propertiesNode.get("LineDelimiter");
                    if (node != null && node.isTextual()) {
                        config.setLineDelimiter(node.asText());
                    }
                    node = propertiesNode.get("ColumnDelimiter");
                    if (node != null && node.isTextual()) {
                        config.setColumnDelimiter(node.asText());
                    }
                    node = propertiesNode.get("NullValue");
                    if (node != null && node.isTextual()) {
                        config.setNullValue(node.asText());
                    }
                }
                result = config;
                break;
            default:
                return result;
        }

        node = tree.get("ColumnNames");
        if (node != null && node.isArray()) {
            List<String> columnNames = new ArrayList<>();
            for (JsonNode temp : node) {
                columnNames.add(temp.asText());
            }
            result.setColumnNames(columnNames);
        }

        return result;
    }

    private void deserializeCommonConfig(ConnectorConfig config, JsonNode tree) {
        JsonNode node = tree.get("TimestampUnit");
        if (node != null && node.isTextual()) {
            config.setTimestampUnit(ConnectorConfig.TimestampUnit.valueOf(node.asText().toUpperCase()));
        }
    }

    private ConnectorConfig deserializeSinkOdpsConfig(JsonNode tree) {
        SinkOdpsConfig config = new SinkOdpsConfig();
        JsonNode node = tree.get("Project");
        if (node != null && node.isTextual()) {
            config.setProject(node.asText());
        }
        node = tree.get("Table");
        if (node != null && node.isTextual()) {
            config.setTable(node.asText());
        }
        node = tree.get("AccessId");
        if (node != null && node.isTextual()) {
            config.setAccessId(node.asText());
        }
        node = tree.get("OdpsEndpoint");
        if (node != null && node.isTextual()) {
            config.setEndpoint(node.asText());
        }
        node = tree.get("TunnelEndpoint");
        if (node != null && node.isTextual()) {
            config.setTunnelEndpoint(node.asText());
        }
        node = tree.get("PartitionMode");
        if (node != null && node.isTextual()) {
            config.setPartitionMode(SinkOdpsConfig.PartitionMode.valueOf(node.asText().toUpperCase()));
        }
        node = tree.get("TimeRange");
        if (node != null && node.isTextual()) {
            config.setTimeRange(Integer.valueOf(node.asText()));
        }
        node = tree.get("TimeZone");
        if (node != null && node.isTextual()) {
            config.setTimeZone(node.asText());
        }

        {
            node = tree.get("SinkCompressData");
            if (node != null && node.isTextual()) {
                config.getInternalConfig().setSinkCompressData(Boolean.valueOf(node.asText()));
            }
            node = tree.get("AddTtHostLine");
            if (node != null && node.isTextual()) {
                config.getInternalConfig().setAddTtHostLine(Boolean.valueOf(node.asText()));
            }
            node = tree.get("AplusDataType");
            if (node != null && node.isTextual()) {
                config.getInternalConfig().setAplusDataType(Boolean.valueOf(node.asText()));
            }
        }

        node = tree.get("Base64Encode");
        if (node != null && node.isTextual()) {
            config.setBase64Encode(Boolean.valueOf(node.asText()));
        }
        node = tree.get("SplitKey");
        if (node != null && node.isTextual()) {
            config.setSplitKey(node.asText());
        }
        node = tree.get("PartitionConfig");
        if (node != null && node.isTextual()) {
            node = JsonUtils.toJsonNode(node.asText());
            if (node != null && node.isArray()) {
                SinkOdpsConfig.PartitionConfig partitionConfig = new SinkOdpsConfig.PartitionConfig();
                for (JsonNode fNode : node) {
                    JsonNode temp = fNode.get("key");
                    String key = temp.asText();
                    temp = fNode.get("value");
                    String value = temp.asText();
                    partitionConfig.addConfig(key, value);
                }
                config.setPartitionConfig(partitionConfig);
            }
        }
        return config;
    }

    private ConnectorConfig deserializeSinkDatabaseConfig(JsonNode tree) {
        SinkMysqlConfig config = new SinkMysqlConfig();
        JsonNode node = tree.get("Host");
        if (node != null && node.isTextual()) {
            config.setHost(node.asText());
        }
        node = tree.get("Port");
        if (node != null && node.isTextual()) {
            config.setPort(Integer.valueOf(node.asText()));
        }
        node = tree.get("Database");
        if (node != null && node.isTextual()) {
            config.setDatabase(node.asText());
        }
        node = tree.get("User");
        if (node != null && node.isTextual()) {
            config.setUser(node.asText());
        }
        node = tree.get("Table");
        if (node != null && node.isTextual()) {
            config.setTable(node.asText());
        }

        node = tree.get("Ignore");
        if (node != null && node.isTextual()) {
            config.setInsertMode(Boolean.valueOf(node.asText()) ?
                    SinkMysqlConfig.InsertMode.IGNORE : SinkMysqlConfig.InsertMode.OVERWRITE);
        }
        return config;
    }

    private void deserializeSinkDatahubConfig(JsonNode tree, SinkDatahubConfig config) {
        JsonNode node = tree.get("Endpoint");
        if (node != null && node.isTextual()) {
            config.setEndpoint(node.asText());
        }
        node = tree.get("Project");
        if (node != null && node.isTextual()) {
            config.setProjectName(node.asText());
        }
        node = tree.get("Topic");
        if (node != null && node.isTextual()) {
            config.setTopicName(node.asText());
        }
        node = tree.get("AuthMode");
        if (node != null && node.isTextual()) {
            config.setAuthMode(SinkConfig.AuthMode.valueOf(node.asText().toUpperCase()));
        }
        node = tree.get("AccessId");
        if (node != null && node.isTextual()) {
            config.setAccessId(node.asText());
        }
    }

    private ConnectorConfig deserializeSinkDatahubConfig(JsonNode tree) {
        SinkDatahubConfig config = new SinkDatahubConfig();
        deserializeSinkDatahubConfig(tree, config);
        return config;
    }

    private ConnectorConfig deserializeSinkHologresConfig(JsonNode tree) {
        SinkHologresConfig config = new SinkHologresConfig();
        deserializeSinkDatahubConfig(tree, config);
        JsonNode node = tree.get("InstanceId");
        if (node != null && node.isTextual()) {
            config.setInstanceId(node.asText());
        }
        node = tree.get("BinaryParserConfig");
        if (node != null && node.isTextual()) {
            JsonNode binaryParserConfigNode = JsonUtils.toJsonNode(node.asText());
            if (binaryParserConfigNode != null && binaryParserConfigNode.isObject()) {
                config.setBinaryParserConfig(parseBinaryParserConfig(binaryParserConfigNode));
            }
        }
        return config;
    }

    private ConnectorConfig deserializeSinkEsConfig(JsonNode tree) {
        SinkEsConfig config = new SinkEsConfig();
        JsonNode node = tree.get("Index");
        if (node != null && node.isTextual()) {
            config.setIndex(node.asText());
        }
        node = tree.get("Endpoint");
        if (node != null && node.isTextual()) {
            config.setEndpoint(node.asText());
        }
        node = tree.get("User");
        if (node != null && node.isTextual()) {
            config.setUser(node.asText());
        }
        node = tree.get("IDFields");
        if (node != null && node.isTextual()) {
            JsonNode fNode = JsonUtils.toJsonNode(node.asText());
            if (fNode != null && fNode.isArray()) {
                List<String> idFields = new ArrayList<>();
                for (JsonNode temp : fNode) {
                    idFields.add(temp.asText());
                }
                config.setIdFields(idFields);
            }
        }
        node = tree.get("TypeFields");
        if (node != null && node.isTextual()) {
            JsonNode fNode = JsonUtils.toJsonNode(node.asText());
            if (fNode != null && fNode.isArray()) {
                List<String> typeFields = new ArrayList<>();
                for (JsonNode temp : fNode) {
                    typeFields.add(temp.asText());
                }
                config.setTypeFields(typeFields);
            }
        }

        node = tree.get("RouterFields");
        if (node != null && node.isTextual()) {
            JsonNode fNode = JsonUtils.toJsonNode(node.asText());
            if (fNode != null && fNode.isArray()) {
                List<String> routerFields = new ArrayList<>();
                for (JsonNode temp : fNode) {
                    routerFields.add(temp.asText());
                }
                config.setRouterFields(routerFields);
            }
        }

        node = tree.get("ProxyMode");
        if (node != null && node.isTextual()) {
            config.setProxyMode(Boolean.valueOf(node.asText()));
        }
        return config;
    }

    private ConnectorConfig deserializeSinkFcConfig(JsonNode tree) {
        SinkFcConfig config = new SinkFcConfig();
        JsonNode node = tree.get("Endpoint");
        if (node != null && node.isTextual()) {
            config.setEndpoint(node.asText());
        }
        node = tree.get("AccessId");
        if (node != null && node.isTextual()) {
            config.setAccessId(node.asText());
        }
        node = tree.get("Service");
        if (node != null && node.isTextual()) {
            config.setService(node.asText());
        }
        node = tree.get("Function");
        if (node != null && node.isTextual()) {
            config.setFunction(node.asText());
        }
        node = tree.get("AuthMode");
        if (node != null && node.isTextual()) {
            config.setAuthMode(SinkConfig.AuthMode.valueOf(node.asText().toUpperCase()));
        }
        node = tree.get("InvokeType");
        if (node != null && node.isTextual()) {
            config.setInvokeType(SinkFcConfig.InvokeType.valueOf(node.asText().toUpperCase()));
        }
        return config;
    }

    private ConnectorConfig deserializeSinkOtsConfig(JsonNode tree) {
        SinkOtsConfig config = new SinkOtsConfig();
        JsonNode node = tree.get("Endpoint");
        if (node != null && node.isTextual()) {
            config.setEndpoint(node.asText());
        }
        node = tree.get("AccessId");
        if (node != null && node.isTextual()) {
            config.setAccessId(node.asText());
        }
        node = tree.get("InstanceName");
        if (node != null && node.isTextual()) {
            config.setInstance(node.asText());
        }
        node = tree.get("TableName");
        if (node != null && node.isTextual()) {
            config.setTable(node.asText());
        }
        node = tree.get("WriteMode");
        if (node != null && node.isTextual()) {
            config.setWriteMode(SinkOtsConfig.WriteMode.valueOf(node.asText().toUpperCase()));
        }
        node = tree.get("AuthMode");
        if (node != null && node.isTextual()) {
            config.setAuthMode(SinkConfig.AuthMode.valueOf(node.asText().toUpperCase()));
        }
        return config;
    }

    private ConnectorConfig deserializeSinkOssConfig(JsonNode tree) {
        SinkOssConfig config = new SinkOssConfig();
        JsonNode node = tree.get("Endpoint");
        if (node != null && node.isTextual()) {
            config.setEndpoint(node.asText());
        }
        node = tree.get("AccessId");
        if (node != null && node.isTextual()) {
            config.setAccessId(node.asText());
        }
        node = tree.get("Bucket");
        if (node != null && node.isTextual()) {
            config.setBucket(node.asText());
        }
        node = tree.get("Prefix");
        if (node != null && node.isTextual()) {
            config.setPrefix(node.asText());
        }
        node = tree.get("TimeFormat");
        if (node != null && node.isTextual()) {
            config.setTimeFormat(node.asText());
        }
        node = tree.get("TimeRange");
        if (node != null && node.isTextual()) {
            config.setTimeRange(Integer.valueOf(node.asText()));
        }
        node = tree.get("MaxFileSize");
        if (node != null && node.isTextual()) {
            config.setMaxFileSize(node.asLong());
        }
        node = tree.get("AuthMode");
        if (node != null && node.isTextual()) {
            config.setAuthMode(SinkConfig.AuthMode.valueOf(node.asText().toUpperCase()));
        }
        return config;
    }

    private ConnectorConfig deserializeSourceDtsConfig(JsonNode tree) {
        SourceDtsConfig config = new SourceDtsConfig();
        JsonNode node = tree.get("Endpoint");
        if (node != null && node.isTextual()) {
            config.setEndpoint(node.asText());
        }
        node = tree.get("User");
        if (node != null && node.isTextual()) {
            config.setUser(node.asText());
        }
        node = tree.get("Topic");
        if (node != null && node.isTextual()) {
            config.setTopic(node.asText());
        }
        node = tree.get("SubId");
        if (node != null && node.isTextual()) {
            config.setSubId(node.asText());
        }
        node = tree.get("IDFields");
        if (node != null && node.isTextual()) {
            JsonNode fNode = JsonUtils.toJsonNode(node.asText());
            if (fNode != null && fNode.isArray()) {
                List<String> idFields = new ArrayList<>();
                for (JsonNode temp : fNode) {
                    idFields.add(temp.asText());
                }
                config.setIdFields(idFields);
            }
        }
        return config;
    }
}