/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aliyun.lindorm.pool;

import com.alibaba.druid.pool.DruidDataSourceFactory;

import javax.naming.Context;
import javax.naming.Name;
import javax.naming.spi.ObjectFactory;
import javax.sql.DataSource;
import java.util.Hashtable;
import java.util.Map;
import java.util.Properties;

/**
 * 使用包装模式将Druid的连接池内置在内部，同时通过shade方式将Druid的打包在内部
 * 避免Druid版本冲突而导致最佳实践无法使用的情况出现
 *
 * @author jianhong.hjh
 */
public class LindormDataSourceFactory implements ObjectFactory {

  // 初始化连接池
  public static final String INIT_KEY = "init";
  // 连接初始数
  public static final String INITIAL_SIZE_KEY = "initialSize";
  // 连接最大idle数
  public static final String MIN_IDLE_KEY = "minIdle";
  // 连接最大数
  public static final String MAX_ACTIVE_KEY = "maxActive";
  // 获取连接最大等待时间，单位毫秒
  public static final String MAX_WAIT_KEY = "maxWait";

  // 连接保活配置项, 避免连接断开异常ConnectionDisconnectedException
  public static final String KEEP_ALIVE_KEY = "druid.keepAlive";
  public static final String KEEP_ALIVE_BETWEEN_TIME_MILLIS_KEY = "druid.keepAliveBetweenTimeMillis";
  public static final String MIN_EVICTABLE_IDLE_TIME_MILLIS_KEY = "minEvictableIdleTimeMillis";
  public static final String MAX_EVICTABLE_IDLE_TIME_MILLIS_KEY = "maxEvictableIdleTimeMillis";
  public static final String TIME_BETWEEN_EVICTION_RUNS_MILLIS_KEY = "timeBetweenEvictionRunsMillis";

  // 连接验证配置项
  public static final String VALIDATION_QUERY_KEY = "validationQuery";
  public static final String TEST_WHILE_IDLE_KEY = "testWhileIdle";
  public static final String TEST_ON_BORROW_KEY = "testOnBorrow";
  public static final String TEST_ON_RETURN_KEY = "testOnReturn";

  // PreparedStatement缓存配置项
  public static final String POOL_PREPARED_STATEMENTS_KEY = "poolPreparedStatements";
  public static final String MAX_OPEN_PREPARED_STATEMENTS_KEY = "maxOpenPreparedStatements";
  public static final String MAX_POOL_PREPARED_STATEMENT_PER_CONNECTION_SIZE_KEY = "druid.maxPoolPreparedStatementPerConnectionSize";

  private DruidDataSourceFactory underlying = new DruidDataSourceFactory();

  @Override public Object getObjectInstance(Object obj, Name name, Context nameCtx, Hashtable<?, ?> environment)
      throws Exception {
    return this.underlying.getObjectInstance(obj, name, nameCtx, environment);
  }

  public static DataSource createDataSource(Properties properties) throws Exception {
    return createDataSource((Map) properties);
  }

  public static DataSource createDataSource(Map properties) throws Exception {
    LindormDataSource dataSource = new LindormDataSource();
    // 检查并设置默认值，避免用户侧未设置默认值
    setDefaultValues(properties);
    DruidDataSourceFactory.config(dataSource, properties);
    return dataSource;
  }

  private static void setDefaultValues(Map properties) {
    // 设置连接初始化默认配置
    setInitDefaultValues(properties);
    // 设置连接保活默认配置
    setKeepAliveValues(properties);
    // 设置PreparedStatements默认值
    setPreparedStatementsDefaultValues(properties);
  }

  private static void setInitDefaultValues(Map properties) {
    if (!properties.containsKey(INIT_KEY)) {
      properties.put(INIT_KEY, "true");
    }
    if (!properties.containsKey(INITIAL_SIZE_KEY)) {
      properties.put(INITIAL_SIZE_KEY, "2");
    }
    if (!properties.containsKey(MIN_IDLE_KEY)) {
      properties.put(MIN_IDLE_KEY, "2");
    }
    if (!properties.containsKey(MAX_ACTIVE_KEY)) {
      properties.put(MAX_ACTIVE_KEY, "5");
    }

    if (!properties.containsKey(MAX_WAIT_KEY)) {
      // 获取连接最大等待时间，单位毫秒
      properties.put(MAX_WAIT_KEY, "30000");
    }
  }

  private static void setKeepAliveValues(Map properties) {
    /*  ***********连接保活配置项*************** */
    if (!properties.containsKey(KEEP_ALIVE_KEY)) {
      properties.put(KEEP_ALIVE_KEY, "true");
    }
    if (!properties.containsKey(KEEP_ALIVE_BETWEEN_TIME_MILLIS_KEY)) {
      properties.put(KEEP_ALIVE_BETWEEN_TIME_MILLIS_KEY, "30000");
    }
    if (!properties.containsKey(MIN_EVICTABLE_IDLE_TIME_MILLIS_KEY)) {
      properties.put(MIN_EVICTABLE_IDLE_TIME_MILLIS_KEY, "600000");
    }

    if (!properties.containsKey(MAX_EVICTABLE_IDLE_TIME_MILLIS_KEY)) {
      properties.put(MAX_EVICTABLE_IDLE_TIME_MILLIS_KEY, "900000");
    }

    if (!properties.containsKey(TIME_BETWEEN_EVICTION_RUNS_MILLIS_KEY)) {
      properties.put(TIME_BETWEEN_EVICTION_RUNS_MILLIS_KEY, "5000");
    }

    /*  ***********连接验证配置项*************** */
    if (!properties.containsKey(VALIDATION_QUERY_KEY)) {
      properties.put(VALIDATION_QUERY_KEY, "SELECT 1");
    }

    if (!properties.containsKey(TEST_WHILE_IDLE_KEY)) {
      properties.put(TEST_WHILE_IDLE_KEY, "true");
    }

    if (!properties.containsKey(TEST_ON_BORROW_KEY)) {
      properties.put(TEST_ON_BORROW_KEY, "false");
    }

    if (!properties.containsKey(TEST_ON_RETURN_KEY)) {
      properties.put(TEST_ON_RETURN_KEY, "false");
    }
  }

  private static void setPreparedStatementsDefaultValues(Map properties) {
    if (!properties.containsKey(POOL_PREPARED_STATEMENTS_KEY)) {
      properties.put(POOL_PREPARED_STATEMENTS_KEY, "false");
    }

    if (!properties.containsKey(MAX_OPEN_PREPARED_STATEMENTS_KEY)) {
      properties.put(MAX_OPEN_PREPARED_STATEMENTS_KEY, "-1");
    }

    if (!properties.containsKey(MAX_POOL_PREPARED_STATEMENT_PER_CONNECTION_SIZE_KEY)) {
      properties.put(MAX_POOL_PREPARED_STATEMENT_PER_CONNECTION_SIZE_KEY, "-1");
    }
  }
}
