/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aliyun.lindorm.search.client;

import org.apache.calcite.avatica.BuiltInConnectionProperty;
import org.apache.calcite.avatica.DriverVersion;
import org.apache.calcite.avatica.remote.HADriver;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Properties;

public class Driver extends HADriver {

  public static final String CONNECT_STRING_PREFIX = "jdbc:lindorm:search:";

  public final static String JDBC_PROTOCOL_TERMINATOR = ";";
  public final static String PROPERTY_SEPARATOR = "=";

  private static final String DEFAULT_SERIALIZATION = "PROTOBUF";
  private static final String DEFAULT_ENDPOINT = "/api/v2/avatica";

  static {
    new Driver().register();
  }

  public Driver() {
    super();
  }

  @Override
  protected DriverVersion createDriverVersion() {
    return DriverVersion.load(
        Driver.class,
        "com-aliyun-lindorm-search-jdbc.properties",
        "Lindorm Search JDBC Driver",
        "unknown version",
        "Lindorm Search",
        "unknown version");
  }

  @Override
  protected String getConnectStringPrefix() {
    return CONNECT_STRING_PREFIX;
  }

  @Override
  public Connection connect(String url, Properties userInfo) throws SQLException {
    final Properties info = new Properties(userInfo);
    info.putAll(userInfo);
    if (!acceptsURL(url)) {
      return null;
    }

    if (!hasUsernameAndPassword(info)) {
      throw new SQLException("user and password can't be null");
    }
    String prefix = this.getConnectStringPrefix();
    String urlSuffix = url.substring(prefix.length());
    info.putAll(readExtraConfigFromUrl(urlSuffix));
    final String urlString = info.getProperty(BuiltInConnectionProperty.URL.camelName());
    if (urlString != null && !urlString.contains(DEFAULT_ENDPOINT)) {
      info.put(BuiltInConnectionProperty.URL.camelName(), urlString + DEFAULT_ENDPOINT);
    }
    info.put(BuiltInConnectionProperty.SERIALIZATION.camelName(), DEFAULT_SERIALIZATION);
    setConnectionInfo(info);
    Connection conn = super.connect(CONNECT_STRING_PREFIX, info);
    String schemaName = info.getProperty(BuiltInConnectionProperty.SCHEMA.camelName());
    if (schemaName != null) {
      conn.setSchema(schemaName);
    }
    return conn;
  }


  /**
   * @param url example jdbc:lindorm:search:[key=value[;key=value...]]
   * @return
   */
  private Properties readExtraConfigFromUrl(String url) {
    Properties info = new Properties();
    String[] kvs = url.split(JDBC_PROTOCOL_TERMINATOR);
    for (String kv: kvs){
      String[] fields = kv.split(PROPERTY_SEPARATOR);
      if (fields.length == 2) {
        info.put(fields[0].trim(), fields[1].trim());
      }
    }
    return info;
  }
}
