/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.calcite.avatica.remote;

import org.apache.calcite.avatica.AvaticaConnection;
import org.apache.calcite.avatica.ConnectionConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Properties;

public class HADriver extends Driver {
    private static final Logger LOG = LoggerFactory.getLogger(HADriver.class);
    private Properties connectionInfo = new Properties();

    private static final String FAILOVER_NAME = "failover";
    private static final String DEFULT_FAILOVER_VALUE = "true";
    private static final String MAX_RETRIES_NAME = "maxRetries";
    private static final String INTERVAL_MILLIS_NAME = "interval";
    private static final String MAX_DELAY_NAME = "maxDelay";
    private static final Integer DEFAULT_MAX_RETRIES = 6;
    private static final Integer DEFAULT_MAX_DELAY_BASE = 15000; // ms
    private static final Integer DEFAULT_INTERVAL_MILLIS = 500; // ms

    public static final String LINDORM_TSDB_DRIVER_CONNECT_TIMEOUT = "lindorm.tsdb.driver.connect.timeout";
    public static final String DEFAULT_LINDORM_TSDB_DRIVER_CONNECT_TIMEOUT = "60000"; //ms
    public static final String LINDORM_TSDB_DRIVER_CONNECTION_REQUEST_TIMEOUT = "lindorm.tsdb.driver.connection.request.timeout";
    public static final String DEFAULT_LINDORM_TSDB_DRIVER_CONNECTION_REQUEST_TIMEOUT = "60000"; //ms
    public static final String LINDORM_TSDB_DRIVER_SOCKET_TIMEOUT = "lindorm.tsdb.driver.socket.timeout";
    // by default not timeout, for compatibility， can be adjust by LINDORM_TSDB_DRIVER_SOCKET_TIMEOUT
    public static final String DEFAULT_LINDORM_TSDB_DRIVER_SOCKET_TIMEOUT = "-1"; //ms
    public static final String LINDORM_TSDB_DRIVER_HTTP_COMPRESSION = "lindorm.tsdb.driver.http.compression";
    public static final String DEFAULT_LINDORM_TSDB_DRIVER_HTTP_COMPRESSION = "false"; //ms

    static {
        new HADriver().register();
    }

    public HADriver() {
        super();
    }

    @Override
    Service createService(AvaticaConnection connection, ConnectionConfig config) {
        final Service.Factory metaFactory = config.factory();
        final Service service;
        if (metaFactory != null) {
            service = metaFactory.create(connection);
        } else if (config.url() != null) {
            final AvaticaHttpClient httpClient = getHttpClient(connection, config);
            final Serialization serializationType = getSerialization(config);
            final boolean failover = Boolean.valueOf(connectionInfo.getProperty(FAILOVER_NAME, DEFULT_FAILOVER_VALUE));
            LOG.debug("Instantiating {} service", serializationType);
            switch (serializationType) {
            case JSON:
                service = new RemoteService(httpClient);
                break;
            case PROTOBUF:
                if (failover) {
                    int maxRetries = Integer.valueOf(connectionInfo.getProperty(MAX_RETRIES_NAME, DEFAULT_MAX_RETRIES.toString()));
                    int intervalMills = Integer.valueOf(connectionInfo.getProperty(INTERVAL_MILLIS_NAME, DEFAULT_INTERVAL_MILLIS.toString()));
                    int maxDelay = Integer.valueOf(connectionInfo.getProperty(MAX_DELAY_NAME, DEFAULT_MAX_DELAY_BASE.toString()));
                    FailoverRetryPolicy retryPolicy = new FailoverRetryPolicy(maxRetries, intervalMills, maxDelay);
                    service = new HARemoteProtobufService(connectionInfo, httpClient, new ProtobufTranslationImpl(), retryPolicy);
                } else {
                    service = new RemoteProtobufService(httpClient, new ProtobufTranslationImpl());
                }
                break;
            default:
                throw new IllegalArgumentException("Unhandled serialization type: " + serializationType);
            }
        } else {
            service = new MockJsonService(Collections.<String, String>emptyMap());
        }
        return service;
    }

    public void setConnectionInfo(Properties connectionInfo) {
        this.connectionInfo = connectionInfo;
    }

    protected boolean hasUsernameAndPassword(Properties connectionInfo) {
        return connectionInfo.getProperty("user") != null
            && connectionInfo.getProperty("password") != null;
    }
}
