/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.calcite.avatica.remote;

import com.aliyun.lindorm.table.blob.LindormBlob;
import org.apache.calcite.avatica.ColumnMetaData;
import org.apache.calcite.avatica.proto.Common;
import org.apache.calcite.avatica.util.InputStreamWithLength;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.sql.Blob;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import static com.aliyun.lindorm.utils.Utils.checkArgument;

public class HARemoteProtobufService extends ProtobufService {
    private static final Logger LOG = LoggerFactory.getLogger(HARemoteProtobufService.class);

    private final AvaticaHttpClient client;
    private final ProtobufTranslation translation;
    private final FailoverRetryPolicy retryPolicy;

    private static final String BLOB_VALUE_NULL_HEADER = "Blob-Null";


    public HARemoteProtobufService(Properties connectionInfo, AvaticaHttpClient client, ProtobufTranslation translation, FailoverRetryPolicy retryPolicy) {
        if (client instanceof AvaticaCommonsHttpClientImpl) {
            this.client = new LindormHttpClient(connectionInfo, (AvaticaCommonsHttpClientImpl) client);
        } else {
            this.client = client;
        }
        this.translation = translation;
        this.retryPolicy = retryPolicy;
    }

    @Override public Blob getBlob(final ColumnMetaData columnMetaData, final byte[] meta) {
        LindormHttpClient lindormHttpClient = (LindormHttpClient) this.client;
        CloseableHttpResponse response = lindormHttpClient.getBlobResponse(columnMetaData.schemaName, columnMetaData.tableName, meta);
        if (response.getFirstHeader(BLOB_VALUE_NULL_HEADER) != null) {
            try {
                response.close();
            } catch (IOException e) {
                //ignore
            }
            return null;
        }
        return new LindormBlob(response);
    }

    @Override public ExecuteResponse apply(ExecuteRequest request) {
        List<InputStreamWithLength> blobStreams = new ArrayList<>();
        List<Integer> blobSlots = new ArrayList<>();
        for (int i = 0; i < request.parameterValues.size(); i++) {
            TypedValue typedValue = request.parameterValues.get(i);
            if (typedValue.value instanceof InputStreamWithLength) {
                if (!Common.BlobOpType.GET_META.equals(request.statementHandle.blobOpType)) {
                    request.statementHandle.blobOpType = Common.BlobOpType.GET_META;
                }
                blobStreams.add((InputStreamWithLength)  typedValue.value);
                blobSlots.add(i);
                request.parameterValues.set(i, TypedValue.EXPLICIT_NULL);
            }
        }

        ExecuteResponse executeResponse = super.apply(request);
        if (blobSlots.size() == 0) {
            return executeResponse;
        }
        List<BlobMetadata> blobMetadatas = executeResponse.results.get(0).blobMetadatas;
        checkArgument(blobStreams.size() == blobSlots.size(), "blob slot size illegal");
        checkArgument(blobMetadatas.size() == blobStreams.size(), "blob meta size illegal");
        List<byte[]> blobValues = new ArrayList<>();
        LindormHttpClient lindormHttpClient = (LindormHttpClient) this.client;
        for (int i = 0; i < blobMetadatas.size(); i++) {
            byte[] blobValue = lindormHttpClient.sendBlob(blobStreams.get(i), blobMetadatas.get(i));
            blobValues.add(blobValue);
        }
        checkArgument(blobValues.size() == blobMetadatas.size(), "blob value size illegal");
        request.statementHandle.blobOpType = Common.BlobOpType.WRITE_META;
        for (int i = 0; i < blobSlots.size(); i++) {
            request.parameterValues.set(blobSlots.get(i), TypedValue.createBytes(blobValues.get(i)));
        }
        return super.apply(request);
    }

    @Override
    public Response _apply(Request request) {
        final Response resp;
        byte[] response;
        try {
            while (true) {
                try {
                    response = client.send(translation.serializeRequest(request));
                    break;
                } catch (Exception e) {
                    LOG.warn("Retrying remote request " + retryPolicy.getRetries() + " times...", e);
                    if (!retryPolicy.awaitUntilNextRetry(e.getCause())) {
                        throw e;
                    }
                }
            }
        } catch (Exception e) {
            LOG.warn("Failed to execute remote request: {}", request);
            throw new RuntimeException(e);
        }

        try {
            resp = translation.parseResponse(response);
        } catch (IOException e) {
            LOG.warn("Failed to deserialize reponse to {}. '{}'", request,
                    new String(response, StandardCharsets.UTF_8));
            // Not a protobuf that we could parse.
            throw new RuntimeException(e);
        }

        // The server had an error, throw an Exception for that.
        if (resp instanceof ErrorResponse) {
            throw ((ErrorResponse) resp).toException();
        }

        return resp;
    }
}
