/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.calcite.avatica.remote;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.calcite.avatica.util.InputStreamWithLength;
import org.apache.http.NoHttpResponseException;
import org.apache.http.client.AuthCache;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.util.EntityUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Base64;
import java.util.Properties;
import java.util.zip.GZIPOutputStream;

public class LindormHttpClient implements AvaticaHttpClient {
    private static final Logger LOG = LoggerFactory.getLogger(LindormHttpClient.class);

    private Properties connectionInfo = null;
    private AvaticaCommonsHttpClientImpl client = null;
    private final int connectTimeout;
    private final int connectionRequestTimeout;
    private final int socketTimeout;
    private final boolean httpCompression;

    private final RequestConfig requestConfig;
    private final URI blobURL;

    private final static ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private static final String METADATA_HEADER = "metadata";
    private static final String DATABASE_HEADER = "database";
    private static final String TABLE_HEADER = "table";
    private static final String PROPERTIES_HEADER = "properties";


    public LindormHttpClient(Properties connectionInfo, AvaticaCommonsHttpClientImpl client) {
        this.connectionInfo = connectionInfo;
        this.client = client;
        connectTimeout = Integer.parseInt(connectionInfo.getProperty(HADriver.LINDORM_TSDB_DRIVER_CONNECT_TIMEOUT,
            HADriver.DEFAULT_LINDORM_TSDB_DRIVER_CONNECT_TIMEOUT));
        connectionRequestTimeout = Integer.parseInt(
            connectionInfo.getProperty(HADriver.LINDORM_TSDB_DRIVER_CONNECTION_REQUEST_TIMEOUT, HADriver.DEFAULT_LINDORM_TSDB_DRIVER_CONNECTION_REQUEST_TIMEOUT));
        socketTimeout = Integer.parseInt(connectionInfo.getProperty(HADriver.LINDORM_TSDB_DRIVER_SOCKET_TIMEOUT,
            HADriver.DEFAULT_LINDORM_TSDB_DRIVER_SOCKET_TIMEOUT));
        httpCompression = Boolean.parseBoolean(connectionInfo.getProperty(HADriver.LINDORM_TSDB_DRIVER_HTTP_COMPRESSION,
            HADriver.DEFAULT_LINDORM_TSDB_DRIVER_HTTP_COMPRESSION));
        requestConfig = RequestConfig.custom().setConnectTimeout(connectTimeout).setConnectionRequestTimeout(connectionRequestTimeout).setSocketTimeout(socketTimeout).build();
        try {
            blobURL = new URI(String.format("http://%s:%d/blob", this.client.uri.getHost(), this.client.uri.getPort()));
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private HttpClientContext createContext() {
        HttpClientContext context = HttpClientContext.create();
        context.setTargetHost(client.host);
        if (null != client.credentials) {
            context.setCredentialsProvider(client.credentialsProvider);
            context.setAuthSchemeRegistry(client.authRegistry);
            AuthCache authCache = new BasicAuthCache();
            authCache.put(client.host, new BasicScheme());
            context.setAuthCache(authCache);
        }
        return context;
    }

    private HttpPost createPost() {
        HttpPost post = new HttpPost(client.uri);
        post.setConfig(requestConfig);
        return post;
    }

    private HttpPost createBlobPost() {
        HttpPost post = new HttpPost(blobURL);
        post.setConfig(requestConfig);
        return post;
    }

    private byte[] execute(HttpPost post, HttpClientContext context) {
        try (CloseableHttpResponse response = this.client.execute(post, context)) {
            final int statusCode = response.getStatusLine().getStatusCode();
            if (HttpURLConnection.HTTP_OK == statusCode || HttpURLConnection.HTTP_INTERNAL_ERROR == statusCode) {
                return EntityUtils.toByteArray(response.getEntity());
            } else if (HttpURLConnection.HTTP_UNAVAILABLE == statusCode) {
                LOG.debug("Failed to connect to server (HTTP/503), retrying");
                // continue
            } else {
                throw new RuntimeException(
                    "Failed to execute HTTP Request, got HTTP/" + statusCode);
            }
        } catch (NoHttpResponseException e) {
            // This can happen when sitting behind a load balancer and a backend server dies
            LOG.debug("The server failed to issue an HTTP response, retrying", e);
            // continue
        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            LOG.debug("Failed to execute HTTP request", e);
            throw new RuntimeException(e);
        }
        return null;
    }

    public byte[] send(byte[] request) {
        HttpClientContext context = createContext();

        if (this.httpCompression) {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            try (GZIPOutputStream gzipOut = new GZIPOutputStream(baos)) {
                gzipOut.write(request);
            } catch (IOException e) {
                throw new RuntimeException("failed to compress requset body");
            }
            request = baos.toByteArray();
        }

        ByteArrayEntity entity = new ByteArrayEntity(request, ContentType.APPLICATION_OCTET_STREAM);

        // Create the client with the AuthSchemeRegistry and manager
        HttpPost post = createPost();
        post.setEntity(entity);

        while (true) {
            byte[] r = execute(post, context);
            if (r != null) {
                return r;
            }
        }
    }

    public byte[] sendBlob(InputStreamWithLength inputStreamWithLength, Service.BlobMetadata blobMetadata) {
        blobMetadata.setLength(inputStreamWithLength.getLength());
        HttpClientContext context = createContext();
        HttpPost post = createBlobPost();
        String metadata;
        String properties;
        try {
            metadata = OBJECT_MAPPER.writeValueAsString(blobMetadata);
            properties = OBJECT_MAPPER.writeValueAsString(connectionInfo);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
        post.setHeader(METADATA_HEADER, metadata);
        post.setHeader(PROPERTIES_HEADER, properties);
        InputStreamEntity inputStreamEntity = new InputStreamEntity(inputStreamWithLength.getInputStream());
        post.setEntity(inputStreamEntity);
        try (CloseableHttpResponse response = this.client.execute(post, context)) {
            final int statusCode = response.getStatusLine().getStatusCode();
            if (HttpURLConnection.HTTP_OK == statusCode) {
                return EntityUtils.toByteArray(response.getEntity());
            } else {
                throw new RuntimeException(
                    "Failed to execute HTTP Request, got HTTP/" + statusCode + "\n" + EntityUtils.toString(response.getEntity()));
            }
        } catch (ClientProtocolException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public CloseableHttpResponse getBlobResponse(String database, String table, byte[] meta) {
        HttpClientContext context = createContext();
        HttpGet get = new HttpGet(blobURL);
        get.setConfig(requestConfig);
        String properties;
        try {
            properties = OBJECT_MAPPER.writeValueAsString(connectionInfo);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
        get.setHeader(METADATA_HEADER, Base64.getEncoder().encodeToString(meta));
        get.setHeader(PROPERTIES_HEADER, properties);
        get.setHeader(DATABASE_HEADER, database);
        get.setHeader(TABLE_HEADER, table);
        try {
            CloseableHttpResponse response = this.client.getHttpClient().execute(get, context);
            int statusCode = response.getStatusLine().getStatusCode();
            if ( statusCode == HttpURLConnection.HTTP_OK) {
                return response;
            } else {
                try {
                    throw new RuntimeException(
                        "Failed to execute HTTP Request, got HTTP/" + statusCode + "\n" + EntityUtils.toString(response.getEntity()));
                } finally {
                    response.close();
                }
            }
        } catch (ClientProtocolException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
