package com.aliyun.openservices.ons.api;

import com.aliyun.openservices.ons.api.spi.InvocationContext;
import com.aliyun.openservices.shade.com.google.common.base.Optional;
import com.aliyun.openservices.shade.com.google.common.base.Strings;

import java.util.List;

public class ONSUnitUtils {
    private static final int FLAG_UNIT = 0x1 << 0;

    //--------多活消息特有属性键-----------
    private static final String MSG_PROP_UNIT_ORIUNIT     = "MSHA.UNIT.ORIUNIT";
    private static final String MSG_PROP_UNIT_TYPE        = "MSHA.UNIT.TYPE";
    private static final String MSG_PROP_UNIT_KEY         = "MSHA.UNIT.KEY";

    //--------消息的系统标识，MessageConst中的系统属性-----------
    private static final String PROPERTY_TRANSIENT_TOPIC_CONFIG = "__RMQ.TRANSIENT.TOPIC_SYS_FLAG";
    private static final String PROPERTY_TRANSIENT_GROUP_CONFIG = "__RMQ.TRANSIENT.GROUP_SYS_FLAG";
    private static final String PROPERTY_TRANSIENT_MSHA_RETRY = "__RMQ.TRANSIENT.MSHA_RETRY";
    private static final String PROPERTY_RETRY_TOPIC = "RETRY_TOPIC";


    //--------消息过滤器上下文属性键-----------
    public static final String CONTEXT_ATTR_TOPIC_UNIT_FLAG = "CONTEXT.TOPIC_UNIT_FLAG";

    /**
     * 多活架构下，获得该多活消息的单元
     * @param msg
     * @return
     */
    public static String getUnitOriUnit(Message msg) {
        return msg.getUserProperties(MSG_PROP_UNIT_ORIUNIT);
    }

    /**
     * 多活架构下，获得消息的单元类型属性
     * @param msg
     * @return
     */
    public static String getUnitType(Message msg) {
        return msg.getUserProperties(MSG_PROP_UNIT_TYPE);
    }

    /**
     * 多活架构下，获得消息的流量标识属性，比如用户ID值等
     * @param msg
     * @return
     */
    public static String getUnitKey(Message msg) {
        return msg.getUserProperties(MSG_PROP_UNIT_KEY);
    }

    /**
     * 多活架构下，设置消息的流量信息
     *
     * @param msg
     * @param oriUnitId
     * @param unitType
     * @param unitKey
     */
    public static void setUnitProperties(Message msg, String oriUnitId, String unitType, String unitKey) {
        msg.putUserProperties(MSG_PROP_UNIT_ORIUNIT, oriUnitId);
        msg.putUserProperties(MSG_PROP_UNIT_TYPE, unitType);
        msg.putUserProperties(MSG_PROP_UNIT_KEY, unitKey);
    }

    // -------------------

    /**
     * 在消息过滤器中，判断消息主题是否是多活主题
     * @return
     */
    public static boolean isUnitTopic(InvocationContext context) {
        Boolean contextTopicFlag = (Boolean) context.getAttributes().get(CONTEXT_ATTR_TOPIC_UNIT_FLAG);
        // 优先从消息属性中判断（消费消息的场景时用）
        Optional<List<Message>> messages = context.getMessages();
        if (!messages.isPresent() || messages.get().size() == 0) {
            return contextTopicFlag == null ? false : contextTopicFlag;
        }
        String msgTopicFlag = MessageAccessor.getSystemProperties(messages.get().get(0), PROPERTY_TRANSIENT_TOPIC_CONFIG);
        if (msgTopicFlag != null && msgTopicFlag.length() > 0) {
            return hasUnitFlag(Integer.parseInt(msgTopicFlag));
        }
        // 其次从当前chain上下文判断（发送消息的场景时用）
        return contextTopicFlag == null ? false : contextTopicFlag;
    }

    /**
     * 在消息过滤器中，判断订阅者是否是打了单元标
     * @return
     */
    public static boolean isUnitGroup(InvocationContext context) {
        // 仅从消息属性中判断（消费消息的场景时用）
        Optional<List<Message>> messages = context.getMessages();
        if (!messages.isPresent() || messages.get().size() == 0) {
            return false;
        }
        String msgGroupFlag = MessageAccessor.getSystemProperties(messages.get().get(0), PROPERTY_TRANSIENT_GROUP_CONFIG);
        if (Strings.isNullOrEmpty(msgGroupFlag)) {
            return false;
        }
        return hasUnitFlag(Integer.parseInt(msgGroupFlag));
    }

    /**
     * 发送消息前对topic打标，ons调用
     */
    public static void setUnitTopic(InvocationContext context, boolean isUnitTopic) {
        context.getAttributes().put(CONTEXT_ATTR_TOPIC_UNIT_FLAG, isUnitTopic);
    }

    /**
     * 是否是重试消息
     */
    public static boolean isRetryMsg(Message message) {
        return !Strings.isNullOrEmpty(message.getSystemProperties(PROPERTY_RETRY_TOPIC));
    }

    /**
     * 设置重试消息需要同步的标记
     */
    public static void setMSHAUnitRetry(Message message) {
        message.putSystemProperties(PROPERTY_TRANSIENT_MSHA_RETRY, String.valueOf(Boolean.TRUE));
    }

    public static String getMSHAUnitRetry(Message message) {
        return message.getSystemProperties(PROPERTY_TRANSIENT_MSHA_RETRY);
    }

    private static boolean hasUnitFlag(final int sysFlag) {
        return (sysFlag & FLAG_UNIT) == FLAG_UNIT;
    }
}
