package com.aliyun.openservices.ons.client;

import com.aliyun.openservices.ons.api.Admin;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.exception.ONSClientException;
import com.aliyun.openservices.ons.client.utils.UtilAll;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicBoolean;
import com.aliyun.openservices.ons.shaded.commons.lang3.StringUtils;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.remoting.CredentialsProvider;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.remoting.StaticCredentialsProvider;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.remoting.StsCredentialsProvider;
import com.aliyun.openservices.ons.shaded.org.slf4j.Logger;
import com.aliyun.openservices.ons.shaded.org.slf4j.LoggerFactory;

public abstract class ClientAbstract implements Admin {
    private static final Logger log = LoggerFactory.getLogger(ClientAbstract.class);
    private static final String TRACE_ON = "true";

    protected CredentialsProvider provider;
    protected String namespace = null;
    protected final String nameServerAddr;
    protected final AtomicBoolean started = new AtomicBoolean(false);
    protected final boolean messageTracingEnabled;

    public ClientAbstract(Properties properties) {
        final String namespaceProps = properties.getProperty(PropertyKeyConst.INSTANCE_ID);
        if (StringUtils.isNoneBlank(namespaceProps)) {
            namespace = namespaceProps;
        }

        final String ramRole = properties.getProperty(PropertyKeyConst.RAM_ROLE_NAME);
        if (StringUtils.isNotBlank(ramRole)) {
            log.info("Ram role was set.");
            this.provider = new StsCredentialsProvider(ramRole);
        } else {
            log.debug("Ram role was not set, using AccessKey/SecretKey/SecurityToken instead.");
            final String accessKey = properties.getProperty(PropertyKeyConst.AccessKey);
            final String secretKey = properties.getProperty(PropertyKeyConst.SecretKey);
            final String securityToken = properties.getProperty(PropertyKeyConst.SecurityToken);

            if (StringUtils.isBlank(accessKey)) {
                throw new ONSClientException("AccessKey is blank unexpectedly, please set it.");
            }
            if (StringUtils.isBlank(secretKey)) {
                throw new ONSClientException("SecretKey is blank unexpectedly, please set it.");
            }
            if (StringUtils.isBlank(securityToken)) {
                this.provider = new StaticCredentialsProvider(accessKey, secretKey);
            } else {
                this.provider = new StaticCredentialsProvider(accessKey, secretKey, securityToken);
            }
        }
        final String nameServerAddr = properties.getProperty(PropertyKeyConst.NAMESRV_ADDR);
        if (!UtilAll.validateNameServerEndpoint(nameServerAddr)) {
            throw new ONSClientException("Name server address is illegal.");
        }
        this.nameServerAddr = nameServerAddr;
        this.messageTracingEnabled = Boolean.parseBoolean(properties.getProperty(PropertyKeyConst.MsgTraceSwitch,
                                                                                 TRACE_ON));
    }

    @Override
    public boolean isStarted() {
        return started.get();
    }

    @Override
    public boolean isClosed() {
        return !isStarted();
    }

    @Override
    public void updateCredential(Properties credentialProperties) {
        final String accessKey = credentialProperties.getProperty(PropertyKeyConst.AccessKey);
        final String secretKey = credentialProperties.getProperty(PropertyKeyConst.SecretKey);
        final String securityToken = credentialProperties.getProperty(PropertyKeyConst.SecurityToken);
        if (StringUtils.isBlank(accessKey)) {
            throw new ONSClientException("update credential failed. please set access key.");
        }
        if (StringUtils.isBlank(secretKey)) {
            throw new ONSClientException("update credential failed. please set secret key");
        }
        if (StringUtils.isBlank(securityToken)) {
            this.provider = new StaticCredentialsProvider(accessKey, secretKey);
            return;
        }
        this.provider = new StaticCredentialsProvider(accessKey, secretKey, securityToken);
    }
}
