package com.aliyun.openservices.ons.client.utils;

import com.aliyun.openservices.ons.api.Constants;
import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.MessageAccessor;
import com.aliyun.openservices.ons.api.SystemProperties;
import com.aliyun.openservices.ons.api.TopicPartition;
import com.aliyun.openservices.ons.api.exception.ONSClientException;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.regex.Pattern;
import com.aliyun.openservices.ons.shaded.commons.lang3.StringUtils;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageImpl;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageQueue;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.protocol.SystemAttribute;
import com.aliyun.openservices.ons.shaded.org.slf4j.Logger;
import com.aliyun.openservices.ons.shaded.org.slf4j.LoggerFactory;

public class UtilAll {
    private static final Logger log = LoggerFactory.getLogger(UtilAll.class);

    private static final String INSTANCE_REGEX = "MQ_INST_\\w+_\\w+";

    private static final Pattern NAME_SERVER_ENDPOINT_PATTERN = Pattern.compile("^(\\w+://|).*");
    private static final Pattern NAME_SERVER_ENDPOINT_WITH_NAMESPACE_PATTERN =
            Pattern.compile("^(\\w+://|)" + INSTANCE_REGEX + "\\..*");

    private UtilAll() {
    }

    /**
     * Convert rocketmq's message to ONS message.
     *
     * @param rmqMessageExt rocketmq's message
     * @return ONS message
     */
    public static Message msgConvert(com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageExt rmqMessageExt) {
        final String topic = rmqMessageExt.getTopic();
        if (StringUtils.isAnyBlank(topic)) {
            throw new ONSClientException("Rocketmq's message topic has blank unexpectedly.");
        }
        final SystemProperties systemProperties = new SystemProperties();
        final String tag = rmqMessageExt.getTag();
        final String keys = rmqMessageExt.getKeys();

        systemProperties.setTag(tag);
        systemProperties.setKey(keys);
        systemProperties.setMsgId(rmqMessageExt.getMsgId());
        systemProperties.setShardingKey(rmqMessageExt.getMessageGroup());
        systemProperties.setReconsumeTimes(rmqMessageExt.getReconsumeTimes());
        systemProperties.setBornTimestamp(rmqMessageExt.getBornTimestamp());
        systemProperties.setBornHost(rmqMessageExt.getBornHost());
        systemProperties.setStartDeliverTime(rmqMessageExt.getDeliveryTimestamp());
        systemProperties.setPartitionOffset(rmqMessageExt.getQueueOffset());

        final Properties userProperties = new Properties();
        userProperties.putAll(rmqMessageExt.getUserProperties());
        return MessageAccessor.message(topic, rmqMessageExt.getBody(), systemProperties, userProperties);
    }

    /**
     * Convert ONS message to rocketmq's message.
     *
     * @param message ONS message
     * @return rocketmq's message
     */
    public static com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.Message msgConvert(Message message) {
        if (null == message) {
            throw new ONSClientException("ONS message is null unexpectedly.");
        }
        final String topic = message.getTopic();
        if (StringUtils.isAnyBlank(topic)) {
            throw new ONSClientException("ONS topic has blank, please set it.");
        }
        ConcurrentMap<String, String> userAttribute = new ConcurrentHashMap<String, String>();
        final Properties userProperties = message.getUserProperties();
        for (String propertyName : userProperties.stringPropertyNames()) {
            userAttribute.put(propertyName, userProperties.getProperty(propertyName));
        }
        final byte[] body = message.getBody();
        if (null == body) {
            throw new ONSClientException("ONS message body is empty, please set it.");
        }
        final SystemAttribute systemAttribute = new SystemAttribute();
        systemAttribute.setBornTimeMillis(message.getBornTimestamp());
        final String key = message.getKey();
        if (null != key) {
            systemAttribute.setKey(key);
        }
        final String tag = message.getTag();
        if (null != tag) {
            systemAttribute.setTag(tag);
        }
        systemAttribute.setMessageId(message.getMsgID());
        final long startDeliverTime = message.getStartDeliverTime();
        if (startDeliverTime > 0) {
            systemAttribute.setDeliveryTimeMillis(startDeliverTime);
        }
        final String shardingKey = message.getShardingKey();
        if (null != shardingKey) {
            systemAttribute.setMessageGroup(shardingKey);
        }
        final String bornHost = message.getBornHost();
        if (null != bornHost) {
            systemAttribute.setBornHost(bornHost);
        }
        MessageImpl messageImpl = new MessageImpl(message.getTopic(), systemAttribute, userAttribute, body);
        return new com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.Message(messageImpl);
    }

    public static boolean validateNameServerEndpoint(String endpoint) {
        return StringUtils.isNoneBlank(endpoint) &&
               (NAME_SERVER_ENDPOINT_PATTERN.matcher(endpoint).matches() ||
                NAME_SERVER_ENDPOINT_WITH_NAMESPACE_PATTERN.matcher(endpoint).matches());
    }

    public static TopicPartition convertToPartition(MessageQueue messageQueue) {
        String topic = messageQueue.getTopic();
        String partition =
                messageQueue.getBrokerName() + Constants.TOPIC_PARTITION_SEPARATOR + messageQueue.getQueueId();
        return new TopicPartition(topic, partition);
    }

    public static MessageQueue convertToMq(TopicPartition topicPartition) {
        String topic = topicPartition.getTopic();
        String[] tmp = topicPartition.getPartition().split(Constants.TOPIC_PARTITION_SEPARATOR);
        if (tmp.length != 2) {
            log.warn("Failed to get message queue from topicPartition={}", topicPartition);
            throw new ONSClientException("Failed to get message queue");
        }
        String brokerName = tmp[0];
        int queueId = Integer.parseInt(tmp[1]);
        return new MessageQueue(topic, brokerName, queueId);
    }
}
