/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics;

import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.GlobalMeterProvider;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.common.Clock;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.internal.SystemClock;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.view.InstrumentSelector;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.view.View;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.resources.Resource;
import java.util.HashMap;
import java.util.Map;

/**
 * Builder class for the {@link SdkMeterProvider}. Has fully functional default implementations of
 * all three required interfaces.
 */
public final class SdkMeterProviderBuilder {

  private Clock clock = SystemClock.getInstance();
  private Resource resource = Resource.getDefault();
  private final Map<InstrumentSelector, View> instrumentSelectorViews = new HashMap<InstrumentSelector, View>();

  SdkMeterProviderBuilder() {}

  /**
   * Assign a {@link Clock}.
   *
   * @param clock The clock to use for all temporal needs.
   * @return this
   */
  public SdkMeterProviderBuilder setClock(Clock clock) {
    if (clock == null) {
      throw new NullPointerException("clock");
    }
    this.clock = clock;
    return this;
  }

  /**
   * Assign a {@link Resource} to be attached to all Spans created by Tracers.
   *
   * @param resource A Resource implementation.
   * @return this
   */
  public SdkMeterProviderBuilder setResource(Resource resource) {
    if (resource == null) {
      throw new NullPointerException("resource");
    }
    this.resource = resource;
    return this;
  }

  /**
   * Register a view with the given {@link InstrumentSelector}.
   *
   * <p>Example on how to register a view:
   *
   * <pre>{@code
   * // create a SdkMeterProviderBuilder
   * SdkMeterProviderBuilder meterProviderBuilder = SdkMeterProvider.builder();
   *
   * // create a selector to select which instruments to customize:
   * InstrumentSelector instrumentSelector = InstrumentSelector.builder()
   *   .setInstrumentType(InstrumentType.COUNTER)
   *   .build();
   *
   * // create a specification of how you want the metrics aggregated:
   * AggregatorFactory aggregatorFactory = AggregatorFactory.minMaxSumCount();
   *
   * // register the view with the SdkMeterProviderBuilder
   * meterProviderBuilder.registerView(instrumentSelector, View.builder()
   *   .setAggregatorFactory(aggregatorFactory).build());
   * }</pre>
   *
   * @since 1.1.0
   */
  public SdkMeterProviderBuilder registerView(InstrumentSelector selector, View view) {
    if (selector == null) {
      throw new NullPointerException("selector");
    }
    if (view == null) {
      throw new NullPointerException("view");
    }
    instrumentSelectorViews.put(selector, view);
    return this;
  }

  /**
   * Returns a new {@link SdkMeterProvider} built with the configuration of this {@link
   * SdkMeterProviderBuilder} and registers it as the global {@link
   * com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.MeterProvider}.
   *
   * @see GlobalMeterProvider
   */
  public SdkMeterProvider buildAndRegisterGlobal() {
    SdkMeterProvider meterProvider = build();
    GlobalMeterProvider.set(meterProvider);
    return meterProvider;
  }

  /**
   * Returns a new {@link SdkMeterProvider} built with the configuration of this {@link
   * SdkMeterProviderBuilder}. This provider is not registered as the global {@link
   * com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.MeterProvider}. It is recommended that you register one provider
   * using {@link SdkMeterProviderBuilder#buildAndRegisterGlobal()} for use by instrumentation when
   * that requires access to a global instance of {@link
   * com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.MeterProvider}.
   *
   * @see GlobalMeterProvider
   */
  public SdkMeterProvider build() {
    ViewRegistryBuilder viewRegistryBuilder = ViewRegistry.builder();
    for (Map.Entry<InstrumentSelector, View> entry : instrumentSelectorViews
        .entrySet()) {
      viewRegistryBuilder.addView(entry.getKey(), entry.getValue());
    }
    ViewRegistry viewRegistry = viewRegistryBuilder.build();
    return new SdkMeterProvider(clock, resource, viewRegistry);
  }
}
