/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.aggregator;

import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.common.Labels;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.common.InstrumentationLibraryInfo;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.InstrumentDescriptor;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.data.AggregationTemporality;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.data.LongSumData;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.data.MetricData;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.resources.Resource;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
final class CountAggregator extends AbstractAggregator<Long> {
  private final AggregationTemporality temporality;

  CountAggregator(
      Resource resource,
      InstrumentationLibraryInfo instrumentationLibraryInfo,
      InstrumentDescriptor descriptor,
      AggregationTemporality temporality) {
    super(
        resource,
        instrumentationLibraryInfo,
        descriptor,
        temporality == AggregationTemporality.CUMULATIVE);
    this.temporality = temporality;
  }

  @Override
  public AggregatorHandle<Long> createHandle() {
    return new Handle();
  }

  @Override
  public Long accumulateDouble(double value) {
    return 1L;
  }

  @Override
  public Long accumulateLong(long value) {
    return 1L;
  }

  @Override
  public Long merge(Long a1, Long a2) {
    return a1 + a2;
  }

  @Override
  public MetricData toMetricData(
      Map<Labels, Long> accumulationByLabels,
      long startEpochNanos,
      long lastCollectionEpoch,
      long epochNanos) {
    return MetricData.createLongSum(
        getResource(),
        getInstrumentationLibraryInfo(),
        getInstrumentDescriptor().getName(),
        getInstrumentDescriptor().getDescription(),
        "1",
        LongSumData.create(
            /* isMonotonic= */ true,
            temporality,
            MetricDataUtils.toLongPointList(
                accumulationByLabels,
                temporality == AggregationTemporality.CUMULATIVE
                    ? startEpochNanos
                    : lastCollectionEpoch,
                epochNanos)));
  }

  static final class Handle extends AggregatorHandle<Long> {
    private static final long DEFAULT_VALUE = 0L;

    private final AtomicLong current = new AtomicLong(DEFAULT_VALUE);

    private Handle() {}

    @Override
    protected void doRecordLong(long value) {
      current.getAndAdd(1);
    }

    @Override
    protected void doRecordDouble(double value) {
      current.getAndAdd(1);
    }

    @Override
    protected Long doAccumulateThenReset() {
      return current.getAndSet(DEFAULT_VALUE);
    }
  }
}
