package com.aliyun.openservices.ons.client.rocketmq.impl;

import com.aliyun.openservices.ons.api.Action;
import com.aliyun.openservices.ons.api.ConsumeContext;
import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.MessageSelector;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.batch.BatchConsumer;
import com.aliyun.openservices.ons.api.batch.BatchMessageListener;
import com.aliyun.openservices.ons.api.exception.ONSClientException;
import com.aliyun.openservices.ons.client.rocketmq.PushConsumer;
import com.aliyun.openservices.ons.client.utils.UtilAll;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import com.aliyun.openservices.ons.shaded.commons.lang3.StringUtils;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.consumer.ConsumeStatus;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageExt;
import com.aliyun.openservices.ons.shaded.org.slf4j.Logger;
import com.aliyun.openservices.ons.shaded.org.slf4j.LoggerFactory;

public class BatchConsumerImpl extends PushConsumer implements BatchConsumer {
    private static final Logger log = LoggerFactory.getLogger(BatchConsumerImpl.class);
    private static final int MAX_BATCH_SIZE = 1024;
    private static final int MIN_BATCH_SIZE = 1;
    private final ConcurrentMap<String, BatchMessageListener> subscribeTable;

    public BatchConsumerImpl(final Properties properties) {
        super(properties);

        this.subscribeTable = new ConcurrentHashMap<String, BatchMessageListener>();

        String maxAwaitBatchSizeProp = properties.getProperty(PropertyKeyConst.ConsumeMessageBatchMaxSize);
        if (!StringUtils.isBlank(maxAwaitBatchSizeProp)) {
            int maxAwaitBatchSize = Math.min(MAX_BATCH_SIZE, Integer.parseInt(maxAwaitBatchSizeProp));
            maxAwaitBatchSize = Math.max(MIN_BATCH_SIZE, maxAwaitBatchSize);
            this.defaultMQPushConsumer.setConsumeMessageBatchMaxSize(maxAwaitBatchSize);
        }
        String maxAwaitTimeInSecondsProp =
                properties.getProperty(PropertyKeyConst.BatchConsumeMaxAwaitDurationInSeconds);
        if (!StringUtils.isBlank(maxAwaitTimeInSecondsProp)) {
            long maxAwaitTimeInSeconds = Long.parseLong(maxAwaitTimeInSecondsProp);
            this.defaultMQPushConsumer.setMaxAwaitTimeMillisPerQueue(maxAwaitTimeInSeconds);
        }
    }

    @Override
    public void start() {
        this.defaultMQPushConsumer.registerMessageListener(new BatchMessageListenerImpl());
        log.info("Register message listener successfully.");
        super.start();
    }

    @Override
    public void subscribe(String topic, String subExpression, BatchMessageListener listener) {
        if (StringUtils.isBlank(topic)) {
            throw new ONSClientException("Topic is blank unexpectedly, please set it.");
        }
        if (null == listener) {
            throw new ONSClientException("Listener is null, please set it.");
        }
        subscribeTable.put(topic, listener);
        final MessageSelector selector = MessageSelector.byTag(subExpression);
        super.subscribe(topic, selector);

    }

    @Override
    public void unsubscribe(String topic) {
        if (StringUtils.isBlank(topic)) {
            return;
        }
        this.subscribeTable.remove(topic);
        super.unsubscribe(topic);
    }

    class BatchMessageListenerImpl extends MessageListenerConcurrently {
        @Override
        public ConsumeStatus consume(List<MessageExt> rmqMessageList,
                                     com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.consumer.ConsumeContext consumeContext) {
            List<Message> messageList = new ArrayList<Message>();
            for (MessageExt rmqMessage : rmqMessageList) {
                messageList.add(UtilAll.msgConvert(rmqMessage));
            }
            BatchMessageListener listener = subscribeTable.get(messageList.get(0).getTopic());
            if (null == listener) {
                throw new ONSClientException("BatchMessageListener is null");
            }
            final ConsumeContext context = new ConsumeContext();

            Action action = listener.consume(messageList, context);
            if (action != null) {
                switch (action) {
                    case CommitMessage:
                        return ConsumeStatus.OK;
                    case ReconsumeLater:
                        return ConsumeStatus.ERROR;
                    default:
                        break;
                }
            }
            return null;
        }
    }
}
