package com.aliyun.openservices.ons.client.rocketmq.impl;

import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.MessageSelector;
import com.aliyun.openservices.ons.api.OffsetStore;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.TopicPartition;
import com.aliyun.openservices.ons.api.exception.ONSClientException;
import com.aliyun.openservices.ons.api.order.ConsumeOrderContext;
import com.aliyun.openservices.ons.api.order.MessageOrderListener;
import com.aliyun.openservices.ons.api.order.OrderAction;
import com.aliyun.openservices.ons.api.order.OrderConsumer;
import com.aliyun.openservices.ons.client.rocketmq.PushConsumer;
import com.aliyun.openservices.ons.client.utils.UtilAll;
import com.aliyun.openservices.ons.shaded.com.google.common.base.Optional;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import com.aliyun.openservices.ons.shaded.commons.lang3.StringUtils;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.consumer.ConsumeStatus;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.consumer.listener.MessageListenerOrderly;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageExt;
import com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.message.MessageQueue;
import com.aliyun.openservices.ons.shaded.org.slf4j.Logger;
import com.aliyun.openservices.ons.shaded.org.slf4j.LoggerFactory;

public class OrderConsumerImpl extends PushConsumer implements OrderConsumer {
    private static final Logger log = LoggerFactory.getLogger(OrderConsumerImpl.class);

    private final ConcurrentMap<String, MessageOrderListener> subscribeTable;

    public OrderConsumerImpl(final Properties properties) {
        super(properties);
        this.subscribeTable = new ConcurrentHashMap<String, MessageOrderListener>();
        final String suspendTimeMillisProps = properties.getProperty(PropertyKeyConst.SuspendTimeMillis);
        if (StringUtils.isNoneBlank(suspendTimeMillisProps)) {
            final long suspendTimeMillis = Long.parseLong(suspendTimeMillisProps);
            defaultMQPushConsumer.setFifoConsumptionSuspendTimeMillis(suspendTimeMillis);
        }
    }

    @Override
    public void start() {
        this.defaultMQPushConsumer.registerMessageListener(new MessageListenerImpl());
        log.info("Register order message listener successfully.");
        super.start();
    }

    @Override
    public void subscribe(String topic, String subExpression, MessageOrderListener listener) {
        MessageSelector selector = MessageSelector.byTag(subExpression);
        this.subscribe(topic, selector, listener);
    }

    @Override
    public void subscribe(String topic, MessageSelector selector, MessageOrderListener listener) {
        if (StringUtils.isBlank(topic)) {
            throw new ONSClientException("Topic is blank unexpectedly, please set it.");
        }
        if (null == listener) {
            throw new ONSClientException("Order listener is null, please set it.");
        }
        subscribeTable.put(topic, listener);
        super.subscribe(topic, selector);
    }

    public void setOffsetStore(final OffsetStore offsetStore) {
        if (null == offsetStore) {
            throw new ONSClientException("OffsetStore is null, please set it.");
        }
        this.defaultMQPushConsumer.setOffsetStore(new com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.impl.consumer.OffsetStore() {

            @Override
            public void start() {
                offsetStore.start();
            }

            @Override
            public void shutdown() {
                offsetStore.shutdown();
            }

            @Override
            public void updateOffset(MessageQueue mq, long offset) {
                final TopicPartition partition = UtilAll.convertToPartition(mq);
                offsetStore.updateOffset(partition, offset);
            }

            @Override
            public Optional<Long> readOffset(MessageQueue mq) {
                final TopicPartition partition = UtilAll.convertToPartition(mq);
                return offsetStore.readOffset(partition);
            }
        });
    }

    @Override
    public void rateLimit(String topic, double permitsPerSecond) {
        defaultMQPushConsumer.rateLimit(topic, permitsPerSecond);
    }

    @Override
    public void unsubscribe(String topic) {
        if (StringUtils.isBlank(topic)) {
            return;
        }
        this.subscribeTable.remove(topic);
        super.unsubscribe(topic);
    }

    class MessageListenerImpl extends MessageListenerOrderly {
        @Override
        public ConsumeStatus consume(List<MessageExt> msgList,
                                     com.aliyun.openservices.ons.shaded.org.apache.rocketmq.client.consumer.ConsumeContext consumeContext) {
            final MessageExt rmqMessageExt = msgList.get(0);
            final Message message = UtilAll.msgConvert(rmqMessageExt);
            final String topic = message.getTopic();
            final MessageOrderListener messageListener = OrderConsumerImpl.this.subscribeTable.get(topic);
            if (null == messageListener) {
                throw new ONSClientException("MessageListener is null.");
            }
            final ConsumeOrderContext context = new ConsumeOrderContext();
            final OrderAction action = messageListener.consume(message, context);
            switch (action) {
                case Success:
                    return ConsumeStatus.OK;
                case Suspend:
                default:
                    return ConsumeStatus.ERROR;
            }
        }
    }
}
