/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.common;

import com.aliyun.openservices.ons.shaded.com.google.auto.value.AutoValue;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.trace.Tracer;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

/**
 * Holds information about the instrumentation library specified when creating an instance of {@link
 * Tracer} using the Tracer Provider.
 */
@AutoValue
@Immutable
public abstract class InstrumentationLibraryInfo {
  private static final InstrumentationLibraryInfo EMPTY = create("", null);

  /**
   * Creates a new instance of {@link InstrumentationLibraryInfo}.
   *
   * @param name name of the instrumentation library (e.g., "com.aliyun.openservices.ons.shaded.io.opentelemetry.contrib.mongodb"),
   *     must not be null
   * @param version version of the instrumentation library (e.g., "1.0.0"), might be null
   * @return the new instance
   */
  public static InstrumentationLibraryInfo create(String name, @Nullable String version) {
    if (name == null) {
      throw new NullPointerException("name");
    }
    return new AutoValue_InstrumentationLibraryInfo(name, version);
  }

  /**
   * Returns an "empty" {@code InstrumentationLibraryInfo}.
   *
   * @return an "empty" {@code InstrumentationLibraryInfo}.
   */
  public static InstrumentationLibraryInfo empty() {
    return EMPTY;
  }

  /**
   * Returns the name of the instrumentation library.
   *
   * @return the name of the instrumentation library.
   */
  public abstract String getName();

  /**
   * Returns the version of the instrumentation library, or {@code null} if not available.
   *
   * @return the version of the instrumentation library, or {@code null} if not available.
   */
  @Nullable
  public abstract String getVersion();

  // Package protected ctor to avoid others to extend this class.
  InstrumentationLibraryInfo() {}
}
