/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics;

import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.BoundLongValueRecorder;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.LongValueRecorder;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.LongValueRecorderBuilder;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.common.Labels;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.aggregator.AggregatorHandle;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.BiFunction;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.InstrumentDescriptor;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.InstrumentType;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.InstrumentValueType;

final class LongValueRecorderSdk extends AbstractSynchronousInstrument
    implements LongValueRecorder {

  private LongValueRecorderSdk(
      InstrumentDescriptor descriptor, SynchronousInstrumentAccumulator<?> accumulator) {
    super(descriptor, accumulator);
  }

  @Override
  public void record(long value, Labels labels) {
    AggregatorHandle<?> aggregatorHandle = acquireHandle(labels);
    try {
      aggregatorHandle.recordLong(value);
    } finally {
      aggregatorHandle.release();
    }
  }

  @Override
  public void record(long value) {
    record(value, Labels.empty());
  }

  @Override
  public BoundLongValueRecorder bind(Labels labels) {
    return new BoundInstrument(acquireHandle(labels));
  }

  static final class BoundInstrument implements BoundLongValueRecorder {
    private final AggregatorHandle<?> aggregatorHandle;

    BoundInstrument(AggregatorHandle<?> aggregatorHandle) {
      this.aggregatorHandle = aggregatorHandle;
    }

    @Override
    public void record(long value) {
      aggregatorHandle.recordLong(value);
    }

    @Override
    public void unbind() {
      aggregatorHandle.release();
    }
  }

  static final class Builder
      extends AbstractSynchronousInstrumentBuilder<LongValueRecorderSdk.Builder>
      implements LongValueRecorderBuilder {

    Builder(
        String name,
        MeterProviderSharedState meterProviderSharedState,
        MeterSharedState meterSharedState) {
      super(
          name,
          InstrumentType.VALUE_RECORDER,
          InstrumentValueType.LONG,
          meterProviderSharedState,
          meterSharedState);
    }

    @Override
    Builder getThis() {
      return this;
    }

    @Override
    public LongValueRecorderSdk build() {
      return buildInstrument(
          new BiFunction<InstrumentDescriptor, SynchronousInstrumentAccumulator<?>, LongValueRecorderSdk>() {
            @Override
            public LongValueRecorderSdk apply(InstrumentDescriptor instrumentDescriptor,
                SynchronousInstrumentAccumulator<?> synchronousInstrumentAccumulator) {
              return new LongValueRecorderSdk(instrumentDescriptor, synchronousInstrumentAccumulator);
            }
          });
    }
  }
}
