/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.aggregator;

import com.aliyun.openservices.ons.shaded.com.google.common.util.concurrent.AtomicDouble;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.metrics.common.Labels;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.common.InstrumentationLibraryInfo;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.common.InstrumentDescriptor;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.data.AggregationTemporality;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.data.DoubleSumData;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.metrics.data.MetricData;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.resources.Resource;
import java.util.Map;

final class DoubleSumAggregator extends AbstractSumAggregator<Double> {
  DoubleSumAggregator(
      Resource resource,
      InstrumentationLibraryInfo instrumentationLibraryInfo,
      InstrumentDescriptor descriptor,
      AggregationTemporality temporality) {
    super(resource, instrumentationLibraryInfo, descriptor, temporality);
  }

  @Override
  public AggregatorHandle<Double> createHandle() {
    return new Handle();
  }

  @Override
  public Double accumulateDouble(double value) {
    return value;
  }

  @Override
  Double mergeSum(Double previousAccumulation, Double accumulation) {
    return previousAccumulation + accumulation;
  }

  @Override
  Double mergeDiff(Double previousAccumulation, Double accumulation) {
    return accumulation - previousAccumulation;
  }

  @Override
  public MetricData toMetricData(
      Map<Labels, Double> accumulationByLabels,
      long startEpochNanos,
      long lastCollectionEpoch,
      long epochNanos) {
    return MetricData.createDoubleSum(
        getResource(),
        getInstrumentationLibraryInfo(),
        getInstrumentDescriptor().getName(),
        getInstrumentDescriptor().getDescription(),
        getInstrumentDescriptor().getUnit(),
        DoubleSumData.create(
            isMonotonic(),
            temporality(),
            MetricDataUtils.toDoublePointList(
                accumulationByLabels,
                temporality() == AggregationTemporality.CUMULATIVE
                    ? startEpochNanos
                    : lastCollectionEpoch,
                epochNanos)));
  }

  static final class Handle extends AggregatorHandle<Double> {
    private static final double DEFAULT_VALUE = 0.0;
    private final AtomicDouble current = new AtomicDouble(DEFAULT_VALUE);

    @Override
    protected Double doAccumulateThenReset() {
      return this.current.getAndSet(DEFAULT_VALUE);
    }

    @Override
    protected void doRecordDouble(double value) {
      current.getAndAdd(value);
    }
  }
}
