/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package com.aliyun.openservices.ons.shaded.io.opentelemetry.sdk.trace;

import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.trace.SpanId;
import com.aliyun.openservices.ons.shaded.io.opentelemetry.api.trace.TraceId;
import javax.annotation.concurrent.ThreadSafe;

/** Interface used by the {@link SdkTracer} to generate new {@link SpanId}s and {@link TraceId}s. */
@ThreadSafe
public abstract class IdGenerator {

  /**
   * Returns a {@link IdGenerator} that generates purely random IDs, which is the default for
   * OpenTelemetry.
   *
   * <p>The underlying implementation uses {@link java.util.concurrent.ThreadLocalRandom} for
   * randomness but may change in the future.
   */
  public static IdGenerator random() {
    // note: check borrowed from OkHttp's check for Android.
    if ("Dalvik".equals(System.getProperty("java.vm.name"))) {
      return AndroidFriendlyRandomIdGenerator.INSTANCE;
    }
    return RandomIdGenerator.INSTANCE;
  }

  /**
   * Generates a new valid {@code SpanId}.
   *
   * @return a new valid {@code SpanId}.
   */
  public abstract String generateSpanId();

  /**
   * Generates a new valid {@code TraceId}.
   *
   * @return a new valid {@code TraceId}.
   */
  public abstract String generateTraceId();
}
