package com.alibaba.schedulerx.worker.actor;

import java.util.List;

import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.common.util.IdUtil;
import com.alibaba.schedulerx.protocol.Worker.ContainerBatchReportTaskStatuesRequest;
import com.alibaba.schedulerx.protocol.Worker.ContainerBatchReportTaskStatuesResponse;
import com.alibaba.schedulerx.protocol.Worker.ContainerReportTaskStatusRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.PullTaskFromMasterRequest;
import com.alibaba.schedulerx.protocol.Worker.PullTaskFromMasterResponse;
import com.alibaba.schedulerx.protocol.Worker.WorkerMapTaskRequest;
import com.alibaba.schedulerx.protocol.Worker.WorkerMapTaskResponse;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.master.MapTaskMaster;
import com.alibaba.schedulerx.worker.master.TaskMaster;
import com.alibaba.schedulerx.worker.master.TaskMasterPool;

import akka.actor.UntypedActor;

/**
 * @author yanxun on 2019/1/9.
 */
public class TaskActor extends UntypedActor {
    private TaskMasterPool masterPool = TaskMasterPool.INSTANCE;
    private static final Logger LOGGER = LogFactory.getLogger(TaskActor.class);

    @Override
    public void onReceive(Object obj) throws Throwable {
        if (obj instanceof ContainerReportTaskStatusRequest) {
            handleTaskStatus((ContainerReportTaskStatusRequest)obj);
        } else if (obj instanceof ContainerBatchReportTaskStatuesRequest) {
            handleBatchTaskStatues((ContainerBatchReportTaskStatuesRequest)obj);
        } else if (obj instanceof WorkerMapTaskRequest) {
            handleMapTask((WorkerMapTaskRequest) obj);
        } else if (obj instanceof PullTaskFromMasterRequest) {
            handlePullTasks((PullTaskFromMasterRequest) obj);
        }
    }

    private void handleTaskStatus(ContainerReportTaskStatusRequest request) {
        try {
            TaskMaster taskMaster = masterPool.get(request.getJobInstanceId());
            LOGGER.debug("handleTaskStatus, uniqueId:{}, status:{}, workerAddr:{}",
                IdUtil.getUniqueId(request.getJobId(), request.getJobInstanceId(), request.getTaskId()),
                request.getStatus(), request.getWorkerAddr());
            if (taskMaster != null) {
                taskMaster.updateTaskStatus(request);
            }
        } catch (Throwable e) {
            LOGGER.error("jobInstanceId={}, taskId={}", request.getJobInstanceId(), request.getTaskId());
        }
    }

    private void handleBatchTaskStatues(ContainerBatchReportTaskStatuesRequest request) {
        LOGGER.info("jobInstanceId={}, serialNum={}, batch receive task status reqs, size:{}", request.getJobInstanceId(), request.getSerialNum(), request.getTaskStatuesCount());
        ContainerBatchReportTaskStatuesResponse response = null;
        try {
            TaskMaster taskMaster = masterPool.get(request.getJobInstanceId());
            if (taskMaster != null) {
                taskMaster.batchUpdateTaskStatus(request);
            }
            response = ContainerBatchReportTaskStatuesResponse.newBuilder()
                .setSuccess(true)
                .setDeliveryId(request.getDeliveryId())
                .build();
        } catch (Throwable e) {
            LOGGER.error("jobInstanceId={}, handleBatchTaskStatues error.", request.getJobInstanceId(), e);
            response = ContainerBatchReportTaskStatuesResponse.newBuilder()
                .setSuccess(false)
                .setMessage(ExceptionUtil.getMessage(e))
                .setDeliveryId(request.getDeliveryId())
                .build();
        } finally {
            getSender().tell(response, getSelf());
        }
    }

    private void handleMapTask(WorkerMapTaskRequest request) throws Exception {
        WorkerMapTaskResponse response = null;
        try {
            long jobInstanceId = request.getJobInstanceId();
            TaskMaster taskMaster = masterPool.get(jobInstanceId);
            if (taskMaster != null) {
                if (!(taskMaster instanceof MapTaskMaster)) {
                    response = WorkerMapTaskResponse.newBuilder().setSuccess(false)
                        .setMessage("TaskMaster is not MapTaskMaster")
                        .build();
                    taskMaster.updateNewInstanceStatus(taskMaster.getSerialNum(), InstanceStatus.FAILED, "TaskMaster is not MapTaskMaster");
                } else {
                    try {
                        long startTime = System.currentTimeMillis();
                        boolean overload = ((MapTaskMaster)taskMaster).map(request.getTaskBodyList(), request.getTaskName());
                        LOGGER.debug("jobInstanceId={} map, cost={}ms", jobInstanceId, (System.currentTimeMillis() - startTime));
                        response = WorkerMapTaskResponse.newBuilder()
                            .setSuccess(true)
                            .setOverload(overload)
                            .build();
                    } catch (Exception e) {
                        LOGGER.error("jobInstanceId={} map error", e);
                        taskMaster.updateNewInstanceStatus(taskMaster.getSerialNum(), InstanceStatus.FAILED, ExceptionUtil.getMessage(e));
                        throw e;
                    }
                }
            } else {
                response = WorkerMapTaskResponse.newBuilder().setSuccess(false)
                    .setMessage("can't found TaskMaster by jobInstanceId=" + jobInstanceId)
                    .build();
            }
        } catch (Throwable e) {
            LOGGER.error("jobInstanceId={}, handleMapTask error.", request.getJobInstanceId(), e);
            response = WorkerMapTaskResponse.newBuilder().setSuccess(false)
                .setMessage(ExceptionUtil.getMessage(e))
                .build();
        } finally {
            getSender().tell(response, getSelf());
        }
    }

    private void handlePullTasks(PullTaskFromMasterRequest request) {
        long jobInstanceId = request.getJobInstanceId();
        TaskMaster taskMaster = masterPool.get(jobInstanceId);
        PullTaskFromMasterResponse response = null;
        try {
            if (taskMaster == null || !(taskMaster instanceof MapTaskMaster)) {
                response = PullTaskFromMasterResponse.newBuilder()
                        .setSuccess(false)
                        .setMessage("TaskMaster is null or not MapTaskMaster, jobInstanceId=" + jobInstanceId)
                        .build();
            } else {
                List<MasterStartContainerRequest> reqs = ((MapTaskMaster)taskMaster).syncPullTasks(request.getSerialNum(),
                        request.getPageSize(), request.getWorkerIdAddr());
                response = PullTaskFromMasterResponse.newBuilder()
                        .setSuccess(true)
                        .addAllRequest(reqs)
                        .build();
            }
        } catch (Throwable e) {
            LOGGER.error("", e);
        } finally {
            getSender().tell(response, getSelf());
        }
    }
}
