package com.alibaba.schedulerx.worker.log.appender;

import java.util.Arrays;
import java.util.Map;

import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.helpers.LogLog;
import org.apache.log4j.spi.LoggingEvent;
import org.apache.log4j.spi.ThrowableInformation;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.SlsInfo;
import com.alibaba.schedulerx.common.util.Base64Util;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.container.ContainerFactory;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.aliyun.openservices.aliyun.log.producer.LogProducer;
import com.aliyun.openservices.aliyun.log.producer.Producer;
import com.aliyun.openservices.aliyun.log.producer.ProducerConfig;
import com.aliyun.openservices.aliyun.log.producer.ProjectConfig;
import com.aliyun.openservices.aliyun.log.producer.errors.ProducerException;
import com.aliyun.openservices.log.common.LogItem;

/**
 * @author xiaomeng.hxm
 *
 */
public class SchedulerxLog4jAppender extends AppenderSkeleton {
    private String project;

    private String endpoint;

    private String accessKeyId;

    private String accessKeySecret;

    private String userAgent = "log4j";

    private String logStore;

    private ProducerConfig producerConfig = new ProducerConfig();

    private String source = "";

    private String timeFormat = "yyyy-MM-dd'T'HH:mm:ssZ";

    private String timeZone = "UTC";

    private Producer producer;

    private DateTimeFormatter formatter;
    
    private LogCollector logCollector = LogCollectorFactory.get();
    
    @Override
    public void activateOptions() {
        try {
            super.activateOptions();
            formatter = DateTimeFormat.forPattern(timeFormat).withZone(DateTimeZone.forID(timeZone));
            producerConfig.setMaxBlockMs(0);
            producerConfig.setTotalSizeInBytes(20 * 1024 * 1024);
    
            ProjectConfig projectConfig = buildProjectConfig();
            if (projectConfig != null) {
                producer = new LogProducer(producerConfig);
                producer.putProjectConfig(projectConfig);
            }
        } catch (Exception e) {
            LogLog.error("", e);
        }

        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                try {
                    doClose();
                } catch (Exception e) {
                    LogLog.error("Failed to close LoghubAppender.", e);
                }
            }
        });
    }

    @Override
    public void close() {
        try {
            doClose();
        } catch (Exception e) {
            LogLog.error("Failed to close LoghubAppender.", e);
        }
    }

    private void doClose() throws InterruptedException, ProducerException {
        if(producer!=null) {
            producer.close();
        }
    }

    @Override
    public boolean requiresLayout() {
        return true;
    }

    private ProjectConfig buildProjectConfig() {
        Configuration conf = ConfigUtil.getWorkerConfig();
        String logServiceType = conf.getString(CommonConstants.LOG_COLLECTOR_TYPE, CommonConstants.LOG_COLLECTOR_TYPE_DEFAULT);
        if (logServiceType.equalsIgnoreCase("sls")) {
            String logConfig = null;
            String[] logConfigList = conf.getStringArray(CommonConstants.LOG_SERVICE_CONFIG);
            if (logConfigList.length > 1) {
                logConfig = StringUtils.join(logConfigList, ",");
            } else if (logConfigList.length == 1){
                logConfig = logConfigList[0];
            }
            if (StringUtils.isNotEmpty(logConfig) && !logConfig.equalsIgnoreCase("null")) {
                SlsInfo slsInfo = JsonUtil.fromJson(logConfig, SlsInfo.class);
                endpoint = slsInfo.getEndpoint();
                project = slsInfo.getProject();
                logStore = slsInfo.getLogstore();
                
                if (conf.getBoolean(CommonConstants.SLS_AKSK_ENCODED, false)) {
                    String encodedAccessKeyId = conf.getString(CommonConstants.SLS_AK);
                    accessKeyId = Base64Util.decode(encodedAccessKeyId);
                    String encodedAccessKeySecret = conf.getString(CommonConstants.SLS_SK);
                    accessKeySecret = Base64Util.decode(encodedAccessKeySecret);
                } else {
                    accessKeyId = conf.getString(CommonConstants.SLS_AK);
                    accessKeySecret = conf.getString(CommonConstants.SLS_SK);
                }
                if (project != null && logStore != null && endpoint != null && accessKeyId != null && accessKeySecret !=null) {
                    LogConfig.INSTANCE.setEnable(true);
                    return new ProjectConfig(project, endpoint, accessKeyId, accessKeySecret);
                }
            }
        }
        
        return null;
    }

    @Override
    protected void append(LoggingEvent event) {
        JobContext jobContext = ContainerFactory.getContainerPool().getContext();
        //如果不在ThreadContainer的线程中，当前版本不收集
        if (jobContext == null) {
            LogLog.warn("jobContext is null, threadName=" + event.getThreadName() + ", message=" + event.getMessage().toString());
            return;
        }
        
        if (producer == null) {
            ProjectConfig projectConfig = buildProjectConfig();
            if (projectConfig != null) {
                producer = new LogProducer(producerConfig);
                producer.putProjectConfig(projectConfig);
            }
        }
        if (producer == null) {
            return;
        }
        
        if (!LogConfig.INSTANCE.isEnable()) {
            return;
        }
        
        LogItem logItem = new LogItem();
        logItem.SetTime((int)(event.getTimeStamp() / 1000));
        DateTime dateTime = new DateTime(event.getTimeStamp());
        logItem.PushBack("executionId", jobContext.getUniqueId());
        logItem.PushBack("time", dateTime.toString(formatter));
        logItem.PushBack("level", event.getLevel().toString());
        logItem.PushBack("thread", event.getThreadName());
        logItem.PushBack("location", event.getLocationInformation().fullInfo);
        Object message = event.getMessage();
        if (message == null) {
            logItem.PushBack("message", "");
        } else {
            logItem.PushBack("message", event.getMessage().toString());
        }

        String throwable = getThrowableStr(event);
        if (throwable != null) {
            logItem.PushBack("throwable", throwable);
        }

        if (getLayout() != null) {
            logItem.PushBack("log", getLayout().format(event));
        } else {
            logItem.PushBack("log", event.getMessage().toString());
        }

        Map properties = event.getProperties();
        if (properties.size() > 0) {
            Object[] keys = properties.keySet().toArray();
            Arrays.sort(keys);
            for (int i = 0; i < keys.length; i++) {
                logItem.PushBack(keys[i].toString(), properties.get(keys[i])
                    .toString());
            }
        }
        
        String topic = createTopic(jobContext);
        try {
            producer.send(project, logStore, topic, source, logItem,
                new SchedulerxLog4jAppenderCallback(project, logStore, topic, source, logItem));
        } catch (Exception e) {
            LogLog.error("Failed to send log, project=" + project
                    + ", logStore=" + logStore
                    + ", topic=" + topic
                    + ", source=" + source
                    + ", logItem=" + logItem, 
                    e);
        }
    }

    /**
     * createTopic
     * @param jobContext
     * @return
     */
    private String createTopic(JobContext jobContext) {
        return "group-" + jobContext.getAppGroupId();
    }

    private String getThrowableStr(LoggingEvent event) {
        ThrowableInformation throwable = event.getThrowableInformation();
        if (throwable == null) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        boolean isFirst = true;
        for (String s : throwable.getThrowableStrRep()) {
            if (isFirst) {
                isFirst = false;
            } else {
                sb.append(System.getProperty("line.separator"));
            }
            sb.append(s);
        }
        return sb.toString();
    }

    public String getProject() {
        return project;
    }

    public void setProject(String project) {
        this.project = project;
    }

    public String getEndpoint() {
        return endpoint;
    }

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public String getAccessKeyId() {
        return accessKeyId;
    }

    public void setAccessKeyId(String accessKeyId) {
        this.accessKeyId = accessKeyId;
    }

    public String getAccessKeySecret() {
        return accessKeySecret;
    }

    public void setAccessKeySecret(String accessKeySecret) {
        this.accessKeySecret = accessKeySecret;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public void setUserAgent(String userAgent) {
        this.userAgent = userAgent;
    }

    public String getLogStore() {
        return logStore;
    }

    public void setLogStore(String logStore) {
        this.logStore = logStore;
    }

    public int getTotalSizeInBytes() {
        return producerConfig.getTotalSizeInBytes();
    }

    public void setTotalSizeInBytes(int totalSizeInBytes) {
        producerConfig.setTotalSizeInBytes(totalSizeInBytes);
    }

    public long getMaxBlockMs() {
        return producerConfig.getMaxBlockMs();
    }

    public void setMaxBlockMs(long maxBlockMs) {
        producerConfig.setMaxBlockMs(maxBlockMs);
    }

    public int getIoThreadCount() {
        return producerConfig.getIoThreadCount();
    }

    public void setIoThreadCount(int ioThreadCount) {
        producerConfig.setIoThreadCount(ioThreadCount);
    }

    public int getBatchSizeThresholdInBytes() {
        return producerConfig.getBatchSizeThresholdInBytes();
    }

    public void setBatchSizeThresholdInBytes(int batchSizeThresholdInBytes) {
        producerConfig.setBatchSizeThresholdInBytes(batchSizeThresholdInBytes);
    }

    public int getBatchCountThreshold() {
        return producerConfig.getBatchCountThreshold();
    }

    public void setBatchCountThreshold(int batchCountThreshold) {
        producerConfig.setBatchCountThreshold(batchCountThreshold);
    }

    public int getLingerMs() {
        return producerConfig.getLingerMs();
    }

    public void setLingerMs(int lingerMs) {
        producerConfig.setLingerMs(lingerMs);
    }

    public int getRetries() {
        return producerConfig.getRetries();
    }

    public void setRetries(int retries) {
        producerConfig.setRetries(retries);
    }

    public int getMaxReservedAttempts() {
        return producerConfig.getMaxReservedAttempts();
    }

    public void setMaxReservedAttempts(int maxReservedAttempts) {
        producerConfig.setMaxReservedAttempts(maxReservedAttempts);
    }

    public long getBaseRetryBackoffMs() {
        return producerConfig.getBaseRetryBackoffMs();
    }

    public void setBaseRetryBackoffMs(long baseRetryBackoffMs) {
        producerConfig.setBaseRetryBackoffMs(baseRetryBackoffMs);
    }

    public long getMaxRetryBackoffMs() {
        return producerConfig.getMaxRetryBackoffMs();
    }

    public void setMaxRetryBackoffMs(long maxRetryBackoffMs) {
        producerConfig.setMaxRetryBackoffMs(maxRetryBackoffMs);
    }

    public String getSource() {
        return source;
    }

    public void setSource(String source) {
        this.source = source;
    }

    public String getTimeFormat() {
        return timeFormat;
    }

    public void setTimeFormat(String timeFormat) {
        this.timeFormat = timeFormat;
    }

    public String getTimeZone() {
        return timeZone;
    }

    public void setTimeZone(String timeZone) {
        this.timeZone = timeZone;
    }
}
