package com.alibaba.schedulerx.worker.metrics;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.google.common.util.concurrent.ThreadFactoryBuilder;

/**
*
* @author xiaomeng.hxm
*/
public class CgroupMetrics {
    private volatile long lastCpuTime = -1;
    private volatile long lastTotalTime = -1;
    private String cgroupPath;
    private ScheduledExecutorService ses = new ScheduledThreadPoolExecutor(1,
            new ThreadFactoryBuilder().setNameFormat("Schedulerx-CgroupMetrics-thread").build(),
            new ThreadPoolExecutor.DiscardPolicy());
    
    private static CgroupMetrics instance = new CgroupMetrics();
    private CgroupMetrics() {
        this.cgroupPath = ConfigUtil.getWorkerConfig().getString(WorkerConstants.CGROUP_PATH_PREFIX, 
                WorkerConstants.CGROUP_PATH_PREFIX_DEFAULT);
        this.lastCpuTime = getCpuTime();
        this.lastTotalTime = System.nanoTime();
        
        this.ses.scheduleAtFixedRate(new Runnable() {
            
            @Override
            public void run() {
                getCpuUsagePercent();
            }
        }, 5, 5, TimeUnit.SECONDS);
    }
    public static CgroupMetrics getInstance() {
        return instance;
    }
    
    private static final Logger LOGGER = LogFactory.getLogger(CgroupMetrics.class);
    
    public long getCpuTime() {
        FileReader fr = null;
        BufferedReader br = null;
        long cpuTime = -1;
        try {
            fr = new FileReader(cgroupPath + "cpuacct.usage");
            br = new BufferedReader(fr);
            String line = br.readLine();
            cpuTime = Long.parseLong(line);
        } catch (FileNotFoundException e) {
            if (LOGGER != null) {
                LOGGER.warn("file not found: " + cgroupPath + "cpuacct.usage");
            }
        } catch (Exception e) {
            if (LOGGER != null) {
                LOGGER.error("", e);
            }
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                    if (LOGGER != null) {
                        LOGGER.error("", e);
                    }
                }
            }
            if (fr != null) {
                try {
                    fr.close();
                } catch (IOException e) {
                    if (LOGGER != null) {
                        LOGGER.error("", e);
                    }
                }
            }
        }
        return cpuTime;
    }
    
    /**
     * cpuUsagePercent = (currentCpuTime - lastCpuTime)) / (currentTotalTime - lastTotalTime)
     * @return cpuUsagePercent
     */
    public double getCpuUsagePercent() {
        double usage = -1;
        long currentCpuTime = getCpuTime();
        if (currentCpuTime > 0) {
            long currentTotalTime = System.nanoTime();
            usage = ((double)(currentCpuTime - lastCpuTime)) / (currentTotalTime - lastTotalTime);
            lastCpuTime = currentCpuTime;
            lastTotalTime = currentTotalTime;
        }
        return usage;
    }
    
    public int getCpuCores() {
        FileReader fr1 = null;
        BufferedReader br1 = null;
        FileReader fr2 = null;
        BufferedReader br2 = null;
        int cores = -1;
        try {
            fr1 = new FileReader(cgroupPath + "cpu.cfs_quota_us");
            br1 = new BufferedReader(fr1);
            float quota = Float.parseFloat(br1.readLine());
            
            if (quota > 0) {
                fr2 = new FileReader(cgroupPath + "cpu.cfs_period_us");
                br2 = new BufferedReader(fr2);
                float period = Float.parseFloat(br2.readLine());
                cores = (int)Math.ceil(quota / period);
            } else {
                cores = Runtime.getRuntime().availableProcessors();
            }
        } catch (Exception e) {
            cores = Runtime.getRuntime().availableProcessors();
            LOGGER.error("", e);
        } finally {
            if (br1 != null) {
                try {
                    br1.close();
                } catch (IOException e) {
                    LOGGER.error("", e);
                }
            }
            if (fr1 != null) {
                try {
                    fr1.close();
                } catch (IOException e) {
                    LOGGER.error("", e);
                }
            }
            if (br2 != null) {
                try {
                    br2.close();
                } catch (IOException e) {
                    LOGGER.error("", e);
                }
            }
            if (fr2 != null) {
                try {
                    fr2.close();
                } catch (IOException e) {
                    LOGGER.error("", e);
                }
            }
        }
        return cores;
    }
    
}
