package com.alibaba.schedulerx.worker.processor;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;

import org.apache.commons.collections.CollectionUtils;

import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.common.util.HessianUtil;
import com.alibaba.schedulerx.protocol.Worker.WorkerMapTaskRequest;
import com.alibaba.schedulerx.protocol.Worker.WorkerMapTaskResponse;
import com.alibaba.schedulerx.protocol.utils.FutureUtils;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.container.ContainerFactory;
import com.alibaba.schedulerx.worker.discovery.GroupManager;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.master.MapTaskMaster;
import com.alibaba.schedulerx.worker.master.TaskMaster;
import com.alibaba.schedulerx.worker.master.TaskMasterPool;
import com.google.common.collect.Lists;
import com.google.protobuf.ByteString;

import akka.actor.ActorSelection;

/**
 *
 * @author xiaomeng.hxm
 */
public abstract class MapJobProcessor extends JavaProcessor {
    private LogCollector logCollector = LogCollectorFactory.get();
    private static final Logger LOGGER = LogFactory.getLogger(MapJobProcessor.class);

    private static final Integer MAX_RETRY_COUNT = 3;

    /**
     * distribute tasks to all workers.
     * 
     * @param taskList, every element in taskList shoudn't beyond 64KB
     * @param taskName
     * @return
     */
    public ProcessResult map(List<? extends Object> taskList, String taskName) {
        ProcessResult result = new ProcessResult(false);
        JobContext context = ContainerFactory.getContainerPool().getContext();
        ActorSelection masterAkkaSelection = SchedulerxWorker.actorSystem.actorSelection(context.getInstanceMasterActorPath());
        if (masterAkkaSelection == null) {
            String errMsg = "get taskMaster akka path error, path=" + context.getInstanceMasterActorPath();
            LOGGER.error(errMsg);
            result.setResult(errMsg);
            return result;
        }

        if (CollectionUtils.isEmpty(taskList)) {
            result.setResult("task list is empty");
            return result;
        }

        int batchSize = ConfigUtil.getWorkerConfig().getInt(WorkerConstants.WORKER_MAP_PAGE_SIZE,
            WorkerConstants.WORKER_MAP_PAGE_SIZE_DEFAULT);
        int size = taskList.size();
        LOGGER.info("map task list, jobInstanceId={}, taskName={}, size={}, batchSize={}", context.getJobInstanceId(), taskName, size, batchSize);
        int quotient = size / batchSize;
        int remainder = size % batchSize;
        /*
            map taskList in #batchNumber batch, every batch has no more than 3000 tasks;
         */
        int batchNumber = remainder > 0 ? quotient + 1 : quotient;
        List<WorkerMapTaskRequest.Builder> builders = Lists.newArrayList();
        for (int i=0; i<batchNumber; i++) {
            builders.add(WorkerMapTaskRequest.newBuilder());
        }

        int position = 0;
        int maxTaskBodySize = ConfigUtil.getWorkerConfig().getInt(WorkerConstants.TASK_BODY_SIZE_MAX,
                WorkerConstants.TASK_BODY_SIZE_MAX_DEFAULT);
        try {
            for (Object task : taskList) {
                this.checkTaskObject(task);
                int batchIdx = position++ / batchSize;
                byte[] taskBody = HessianUtil.toBytes(task);
                if (taskBody.length > maxTaskBodySize) {
                    throw new IOException("taskBody size more than " + maxTaskBodySize + "B!");
                }
                builders.get(batchIdx).addTaskBody(ByteString.copyFrom(taskBody));
            }

            position = 0;
            for (WorkerMapTaskRequest.Builder builder : builders) {
                builder.setJobId(context.getJobId());
                builder.setJobInstanceId(context.getJobInstanceId());
                builder.setTaskId(context.getTaskId());
                builder.setTaskName(taskName);
                WorkerMapTaskResponse response = null;
                int retryCount = 0;
                try {
                    TaskMaster taskMaster = TaskMasterPool.INSTANCE.get(context.getJobInstanceId());
                    if(taskMaster != null && taskMaster instanceof MapTaskMaster){
                        response = handleMapTask(taskMaster, builder.build());
                    }else {
                        response = (WorkerMapTaskResponse) FutureUtils.awaitResult(masterAkkaSelection, builder.build(), 30);
                    }
                }catch (TimeoutException exception){
                    LOGGER.warn("JobInstanceId={} WorkerMapTaskRequest dispatch timeout.", context.getJobInstanceId(), exception);
                    if(retryCount < MAX_RETRY_COUNT) {
                        Thread.sleep(10 * 1000);
                        masterAkkaSelection = SchedulerxWorker.actorSystem.actorSelection(context.getInstanceMasterActorPath());
                        response = (WorkerMapTaskResponse) FutureUtils.awaitResult(masterAkkaSelection, builder.build(), 30);
                        retryCount++;
                    }else {
                        throw exception;
                    }
                }
                if (!response.getSuccess()) {
                    LOGGER.error(response.getMessage());
                    logCollector.collect(context.getAppGroupId(), context.getUniqueId(), response.getMessage());
                    result.setResult(response.getMessage());
                    return result;
                }
                builders.set(position++, null);
                if (response.hasOverload() && response.getOverload()) {
                    // overload, sleep a while
                    LOGGER.warn("Task Master is busy, sleeping a while {}s...", 10);
                    Thread.sleep(10 * 1000);
                }
            }
            result.setStatus(true);
        } catch (Throwable e) {
            LOGGER.error("JobInstanceId={} WorkerMapTaskRequest dispatch error.", context.getJobInstanceId(), e);
            logCollector.collect(context.getAppGroupId(), context.getUniqueId(), ExceptionUtil.getTrace(e));
            result.setResult(ExceptionUtil.getMessage(e));
        }

        return result;
    }

    /**
     * 子任务对象信息检查
     * @param taskObject
     */
    private void checkTaskObject(Object taskObject){
        JobContext context = ContainerFactory.getContainerPool().getContext();
        boolean isAdvancedVersion = GroupManager.INSTANCE.isAdvancedVersion(context.getGroupId());
        if(isAdvancedVersion && taskObject instanceof BizSubTask) {
            BizSubTask bizSubTask = (BizSubTask) taskObject;
            Map<String, String> labelMap = bizSubTask.labelMap();
            if (labelMap.size() > 3) {
                throw new RuntimeException("label map size can't beyond 3.");
            }
            for (Map.Entry<String,String> entry : labelMap.entrySet()){
                if(entry.getKey().length()> 60 || entry.getValue().length()>180){
                    LOGGER.error("Job instance={} label map<{}, {}> content can't beyond max size(60,180).", context.getJobInstanceId(),
                            entry.getKey().length(), entry.getValue());
                    throw new RuntimeException("label map content can't beyond max size(60,180).");
                }
            }
        }

    }

    private WorkerMapTaskResponse handleMapTask(TaskMaster taskMaster, WorkerMapTaskRequest request) throws Exception {
        WorkerMapTaskResponse response = null;
        try {
            long jobInstanceId = request.getJobInstanceId();
            if (taskMaster != null) {
                if (!(taskMaster instanceof MapTaskMaster)) {
                    response = WorkerMapTaskResponse.newBuilder().setSuccess(false)
                            .setMessage("TaskMaster is not MapTaskMaster")
                            .build();
                    taskMaster.updateNewInstanceStatus(taskMaster.getSerialNum(), InstanceStatus.FAILED, "TaskMaster is not MapTaskMaster");
                } else {
                    try {
                        long startTime = System.currentTimeMillis();
                        boolean overload = ((MapTaskMaster)taskMaster).map(request.getTaskBodyList(), request.getTaskName());
                        LOGGER.debug("jobInstanceId={} map, cost={}ms", jobInstanceId, (System.currentTimeMillis() - startTime));
                        response = WorkerMapTaskResponse.newBuilder()
                                .setSuccess(true)
                                .setOverload(overload)
                                .build();
                    } catch (Exception e) {
                        LOGGER.error("jobInstanceId={} map error", e);
                        taskMaster.updateNewInstanceStatus(taskMaster.getSerialNum(), InstanceStatus.FAILED, ExceptionUtil.getMessage(e));
                        throw e;
                    }
                }
            } else {
                response = WorkerMapTaskResponse.newBuilder().setSuccess(false)
                        .setMessage("can't found TaskMaster by jobInstanceId=" + jobInstanceId)
                        .build();
            }
        } catch (Throwable e) {
            LOGGER.error("jobInstanceId={}, handleMapTask error.", request.getJobInstanceId(), e);
            response = WorkerMapTaskResponse.newBuilder().setSuccess(false)
                    .setMessage(ExceptionUtil.getMessage(e))
                    .build();
        }
        return response;
    }

    protected boolean isRootTask(JobContext context) {
        return context.getTaskName().equals(WorkerConstants.MAP_TASK_ROOT_NAME);
    }
}
