package com.alibaba.schedulerx.worker.service;

import com.alibaba.schedulerx.common.domain.JSONResult;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import org.apache.http.ExceptionLogger;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.MethodNotSupportedException;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.nio.bootstrap.HttpServer;
import org.apache.http.impl.nio.bootstrap.ServerBootstrap;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.BasicAsyncResponseProducer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import org.springframework.beans.factory.annotation.Value;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * WorkerHttpServer
 * @author yaohui
 * @create 2023/8/15 4:08 PM
 **/
public class WorkerHttpServer {

    private WorkerControlService workerControlService = new WorkerControlService();

    public WorkerHttpServer() throws IOException {
        int port = ConfigUtil.getWorkerConfig().getInt(WorkerConstants.HTTP_SERVER_PORT, 51886);
        final IOReactorConfig config = IOReactorConfig.custom()
                .setSoTimeout(15000)
                .setIoThreadCount(1)
                .setTcpNoDelay(true)
                .build();
        final HttpServer server = ServerBootstrap.bootstrap()
                .setListenerPort(port)
                .setServerInfo("SchedulerX-Http-Server/1.1")
                .setIOReactorConfig(config)
                .setSslContext(null)
                .setExceptionLogger(ExceptionLogger.STD_ERR)
                .registerHandler("/schedulerx/worker/*", new WorkerHttpServer.WorkerHttpRequestHandler())
                .create();
        server.start();
    }

    class WorkerHttpRequestHandler implements HttpAsyncRequestHandler<HttpRequest> {

        @Override
        public HttpAsyncRequestConsumer<HttpRequest> processRequest(HttpRequest httpRequest, HttpContext httpContext) throws HttpException, IOException {
            return new BasicAsyncRequestConsumer();
        }

        @Override
        public void handle(HttpRequest request, HttpAsyncExchange httpExchange, HttpContext context) throws HttpException {
            final HttpResponse response = httpExchange.getResponse();
            handleInternal(request, response, context);
            httpExchange.submitResponse(new BasicAsyncResponseProducer(response));
        }

        private void handleInternal(final HttpRequest request, final HttpResponse response, final HttpContext context) throws HttpException {

            final String method = request.getRequestLine().getMethod().toUpperCase(Locale.ENGLISH);
            if (!method.equals("GET") && !method.equals("HEAD") && !method.equals("POST")) {
                throw new MethodNotSupportedException(method + " method not supported");
            }
            final String uri = request.getRequestLine().getUri();
            int endIndex = uri.contains("?")?uri.indexOf("?"):uri.length();
            final String action = uri.substring(uri.lastIndexOf("/")+1, endIndex);
            JSONResult jsonResult;
            switch (action) {
                case "shutdown":
                    // 获取请求的参数
                    Integer modeObject = null;
                    try {
                        URIBuilder builder = new URIBuilder(uri);
                        List<NameValuePair> params = builder.getQueryParams();
                        Map<String, String> paramMap = new HashMap<>();
                        for (NameValuePair param : params) {
                            paramMap.put(param.getName(), param.getValue());
                        }
                        modeObject = paramMap.get("mode") != null? Integer.parseInt(paramMap.get("mode")):null;
                    } catch (Exception e) {}
                    jsonResult = workerControlService.shutdown(modeObject);
                    break;
                case "restart":
                    jsonResult = workerControlService.restart();
                    break;
                case "config":
                    jsonResult = workerControlService.config();
                    break;
                default:
                    jsonResult = JSONResult.geneFailResult("not support action.");
            }
            response.setStatusCode(jsonResult.isSuccess()?HttpStatus.SC_OK:HttpStatus.SC_NOT_IMPLEMENTED);
            final NStringEntity body = new NStringEntity(JsonUtil.toJson(jsonResult), ContentType.APPLICATION_JSON);
            response.setEntity(body);
        }
    }

}
