package com.alibaba.schedulerx.worker.util;

import java.io.BufferedReader;
import java.io.StringReader;
import java.net.HttpURLConnection;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

import javax.net.ssl.SSLContext;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

import com.alibaba.schedulerx.common.domain.AppGroupInfo;
import com.alibaba.schedulerx.common.domain.JSONResult;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.google.common.collect.Lists;
import com.mashape.unirest.http.HttpResponse;
import com.mashape.unirest.http.JsonNode;
import com.mashape.unirest.http.Unirest;

/**
 * console接口获取信息
 *
 * @author 陆昆
 * @create 2019-03-13 下午4:49
 **/
public class ConsoleUtil {
    private static final Logger LOGGER = LogFactory.getLogger(ConsoleUtil.class);

    private static final String APP_GROUP_ID_URL = "/worker/v1/appgroup/getId";

    private static final String APP_GROUP_URL = "/worker/v1/appgroup/get";

    private static final String CONSOLE_LIST_URL = "/worker/v1/meta/consolelist";
    
    /**
     * 忽略ssl认证
     */
    static {
        try {
            SSLContext sslcontext = SSLContexts.custom().loadTrustMaterial(null, new TrustSelfSignedStrategy()).build();
            SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslcontext,
                SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
            CloseableHttpClient httpclient = HttpClients.custom().setSSLSocketFactory(sslsf).build();
            Unirest.setHttpClient(httpclient);
//            Unirest.setTimeouts(5000, 5000);
        } catch (Exception e) {
            LOGGER.error("", e);
        }
    }

    public static AppGroupInfo getAppGroup(String groupId, String appKey, String namespace, String namespaceSource) {
        AppGroupInfo appGroupInfo = null;
        Map<String, Object> parameterMap = new HashMap<>();
        parameterMap.put("groupId", groupId);
        String domain = ConfigUtil.getWorkerConfig().getString(WorkerConstants.WORKER_DOMAIN_NAME);
        if (StringUtils.isNotBlank(domain)) {
            String url;
            if (namespace != null) {
                url = "http://" + domain + APP_GROUP_URL + "?groupId=" + groupId + "&namespace=" + namespace;
                if (StringUtils.isNotBlank(namespaceSource)) {
                    url += "&namespaceSource=" + namespaceSource;
                }
            } else {
                url = "http://" + domain + APP_GROUP_URL + "?groupId=" + groupId;
            }
            try {
                url += "&appKey=" + URLEncoder.encode((appKey == null) ? "" : appKey, "UTF-8");
                HttpResponse<JsonNode> jsonResponse = Unirest.get(url).asJson();
                if (jsonResponse != null) {
                    JSONResult jsonResult = JsonUtil.fromJson(jsonResponse.getBody().toString(), JSONResult.class);
                    if (jsonResult != null) {
                        appGroupInfo = JsonUtil.fromJson(JsonUtil.toJson(jsonResult.getData()), AppGroupInfo.class);
                    }
                }
            } catch (Throwable e) {
                LOGGER.warn("getAppGroup groupid={} error, url={} {}", groupId, url, e.getMessage());
            }
        }
        return appGroupInfo;
    }

    public static long getAppGroupId(String groupId, String appKey, String namespace, String namespaceSource) {
        long appGroupId = 0L;
        Map<String, Object> parameterMap = new HashMap<>();
        parameterMap.put("groupId", groupId);
        String domain = ConfigUtil.getWorkerConfig().getString(WorkerConstants.WORKER_DOMAIN_NAME);
        if (StringUtils.isNotBlank(domain)) {
            String url;
            if (namespace != null) {
                url = "http://" + domain + APP_GROUP_ID_URL + "?groupId=" + groupId + "&namespace=" + namespace;
                if (StringUtils.isNotBlank(namespaceSource)) {
                    url += "&namespaceSource=" + namespaceSource;
                }
            } else {
                url = "http://" + domain + APP_GROUP_ID_URL + "?groupId=" + groupId;
            }
            try {
                url += "&appKey=" + URLEncoder.encode((appKey == null) ? "" : appKey, "UTF-8");
                HttpResponse<JsonNode> jsonResponse = Unirest.get(url).asJson();
                if (jsonResponse != null) {
                    JSONResult jsonResult = JsonUtil.fromJson(jsonResponse.getBody().toString(), JSONResult.class);
                    if (jsonResult != null) {
                        appGroupId = NumberUtils.toLong((String)jsonResult.getData(), 0);
                    }
                }
            } catch (Throwable e) {
                LOGGER.error("getAppGroupId error, url={}", url, e);
                throw new RuntimeException("GetAppGroupId failed.", e);
            }
        }
        return appGroupId;
    }

    /**
     * 从地址服务器获取vip列表
     *
     * @return
     */
    public static List<String> getVipServerList(String addressServerDomain, String port) {
        List<String> serverList = Lists.newArrayList();
        String url = null;
        try {
            url = String.format("http://%s:%s/schedulerx2/consolelist", addressServerDomain,
                port);
            HttpResponse<String> response = Unirest.get(url).asString();

            if (response != null && response.getStatus() == HttpURLConnection.HTTP_OK) {
                BufferedReader reader = new BufferedReader(new StringReader(response.getBody()));
                String line;
                while ((line = reader.readLine()) != null) {
                    if (StringUtils.isNotBlank(line)) {
                        serverList.add(StringUtils.trim(line));
                    }
                }
            }
            return serverList;
        } catch (Exception ex) {
            LOGGER.warn("Fetch domainName from address server error, url={}.", url);
            return serverList;
        }
    }

    private static String getAddressServerPort() {
        String addressPort = System.getProperty(WorkerConstants.ADDRESS_SERVER_PORT);

        if (StringUtils.isBlank(addressPort)) {
            addressPort = (String)ConfigUtil.getWorkerConfig().getProperty(WorkerConstants.ADDRESS_SERVER_PORT);
        }

        if (StringUtils.isBlank(addressPort)) {
            addressPort = System.getenv(WorkerConstants.ADDRESS_SERVER_PORT.replace(".", "_"));
        }

        if (StringUtils.isBlank(addressPort) || Integer.parseInt(addressPort) <= 0) {
            addressPort = "8080";
        }

        ConfigUtil.getWorkerConfig().setProperty(WorkerConstants.ADDRESS_SERVER_PORT, addressPort);

        return addressPort;
    }

    public static String getAddressServerDomain() {
        // set优先
        String domainName = ConfigUtil.getWorkerConfig().getString(WorkerConstants.ADDRESS_SERVER_DOMAIN);
        
        // EDAS应用会通过-D配置
        if (StringUtils.isBlank(domainName)) {
            domainName = System.getProperty(WorkerConstants.ADDRESS_SERVER_DOMAIN);
        }

        if (StringUtils.isBlank(domainName)) {
            domainName = System.getenv(WorkerConstants.ADDRESS_SERVER_DOMAIN.replace(".", "_"));
        }

        if (StringUtils.isNotBlank(domainName)) {
            ConfigUtil.getWorkerConfig().setProperty(WorkerConstants.ADDRESS_SERVER_DOMAIN, domainName);
        }

        return domainName;
    }

    /**
     * 直接从地址服务获取domainName
     *
     * @return
     */
    public static String getDomainFromHttpServer() {
        String addressDomain = getAddressServerDomain();
        if (StringUtils.isBlank(addressDomain)) {
            return StringUtils.EMPTY;
        }

        List<String> serverList = getVipServerList(addressDomain, getAddressServerPort());

        if (CollectionUtils.isNotEmpty(serverList)) {
            int index = 0;
            if (serverList.size() > 1) {
                Random random = new Random();
                index = random.nextInt(serverList.size());
            }
            return serverList.get(index);
        }

        return StringUtils.EMPTY;
    }

    @SuppressWarnings("unchecked")
    public static Map<String, Object> fetchMetaInfoFromConsole(String namespace, String namespaceSource,
                                                               List<String> groupIds) throws Exception {
        String url = null;
        try {
            String domain = ConfigUtil.getWorkerConfig().getString(WorkerConstants.WORKER_DOMAIN_NAME);
            if (StringUtils.isBlank(namespaceSource)) {
                url = "http://" + domain + "/worker/v1/meta/get?namespace=" + namespace + "&groups=" + StringUtils.join(
                    groupIds, ",")+"&appKeys="+ConfigUtil.getWorkerConfig().getString((WorkerConstants.APP_KEY));
            } else {
                url = "http://" + domain + "/worker/v1/meta/get?namespace=" + namespace + "&namespaceSource="
                    + namespaceSource + "&groups=" + StringUtils.join(groupIds, ",")+"&appKeys="+ConfigUtil.getWorkerConfig().getString((WorkerConstants.APP_KEY));
            }
            url += "&encodeKey=true&supportLogService=true";
            HttpResponse<JsonNode> response = Unirest.get(url).asJson();
            if (response != null && response.getStatus() == HttpURLConnection.HTTP_OK) {
                JSONResult jsonResult = JsonUtil.fromJson(response.getBody().toString(), JSONResult.class);
                return (Map<String, Object>)jsonResult.getData();
            }
        } catch (Exception ex) {
            LOGGER.error("fetch metaInfo from console error, url={}", url, ex);
            throw ex;
        }

        return null;
    }
    
    public static String getRandomConsoleIp(String domain) {
        String console = null;
        String url = "http://" + domain + CONSOLE_LIST_URL;
        try {
            HttpResponse<JsonNode> response = Unirest.get(url).asJson();
            if (response != null && response.getStatus() == HttpURLConnection.HTTP_OK) {
                JSONResult jsonResult = JsonUtil.fromJson(response.getBody().toString(), JSONResult.class);
                List<String> consoleList = (List<String>) jsonResult.getData();
                if (CollectionUtils.isNotEmpty(consoleList)) {
                    int index = new Random().nextInt(consoleList.size());
                    console = consoleList.get(index);
                }
            }
        } catch (Exception e) {
            LOGGER.error("get consolelist error, url={}", url, e);
        }
        return console;
    }
}
