package com.alibaba.schedulerx.worker.actor;

import java.util.List;
import java.util.concurrent.ThreadPoolExecutor;

import com.alibaba.fastjson.JSON;
import com.alibaba.schedulerx.common.domain.Metrics;
import com.alibaba.schedulerx.common.domain.TaskDispatchMode;
import com.alibaba.schedulerx.common.monitor.MetricsCollector;
import com.alibaba.schedulerx.protocol.Worker.ContainerCheckZombieRequest;
import com.alibaba.schedulerx.protocol.Worker.ContainerCheckZombieResponse;
import com.alibaba.schedulerx.protocol.Worker.MasterCheckWorkerAliveRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterCheckWorkerAliveResponse;
import com.alibaba.schedulerx.protocol.Worker.WorkerOfflineRequest;
import com.alibaba.schedulerx.protocol.Worker.WorkerOfflineResponse;
import com.alibaba.schedulerx.worker.container.ThreadContainerPool;
import com.alibaba.schedulerx.worker.master.TaskMaster;
import com.alibaba.schedulerx.worker.master.TaskMasterPool;
import com.alibaba.schedulerx.worker.pull.PullManager;
import com.google.common.collect.Lists;

import akka.actor.UntypedActor;

/**
 * @author yanxun on 2019/1/9.
 */
public class WorkerHeartbeatActor extends UntypedActor {
    private TaskMasterPool taskMasterPool = TaskMasterPool.INSTANCE;

    @Override
    public void onReceive(Object obj) throws Throwable {
        if (obj instanceof MasterCheckWorkerAliveRequest) {
            handleCheckWorkerAlive((MasterCheckWorkerAliveRequest) obj);
        } else if (obj instanceof ContainerCheckZombieRequest) {
            handleCheckZombie((ContainerCheckZombieRequest) obj);
        } else if (obj instanceof WorkerOfflineRequest) {
            handleWorkerOffline((WorkerOfflineRequest) obj);
        }
     }

    private void handleWorkerOffline(WorkerOfflineRequest request) {
        TaskMaster taskMaster = TaskMasterPool.INSTANCE.get(request.getJobInstanceId());
        if (taskMaster != null) {
            if (request.getShutdown()) {
                taskMaster.handleWorkerShutdown(request.getWorkerIdAddr(), true);
            } else {
                taskMaster.handleWorkerOffline(request.getWorkerIdAddr());
            }
        }
        WorkerOfflineResponse response = WorkerOfflineResponse.newBuilder().setSuccess(true).build();
        getSender().tell(response, getSelf());
    }

    private void handleCheckWorkerAlive(MasterCheckWorkerAliveRequest request) {
        MasterCheckWorkerAliveResponse response;
        long jobInstanceId = request.getJobInstanceId();
        if (request.hasDispatchMode() &&
                request.getDispatchMode().equals(TaskDispatchMode.PULL.getValue()) &&
                PullManager.INSTANCE.contains(jobInstanceId) &&
                PullManager.INSTANCE.isCrashed(jobInstanceId)) {
            response = MasterCheckWorkerAliveResponse.newBuilder()
                    .setSuccess(false)
                    .setMessage(request.getJobInstanceId() + "  is crashed in PullMananger")
                    .build();
        } else {
            Metrics metrics = MetricsCollector.getMetrics();
            ThreadPoolExecutor sharedThreadPool = ThreadContainerPool.getInstance().getSharedThreadPool();
            if (sharedThreadPool != null && metrics != null) {
                // 可用大小可用线程数 + 线程数等量缓冲区
                Integer availableSize = (sharedThreadPool.getCorePoolSize() - sharedThreadPool.getActiveCount()) +
                        ((int)Math.sqrt(sharedThreadPool.getCorePoolSize()) - sharedThreadPool.getQueue().size());
                metrics.setSharePoolAvailableSize(availableSize);
            }
            response = MasterCheckWorkerAliveResponse.newBuilder().setSuccess(true)
                    .setMetricsJson(metrics!=null? JSON.toJSONString(metrics):"")
                    .build();
        }
        getSender().tell(response, getSelf());
    }

    private void handleCheckZombie(ContainerCheckZombieRequest request) {
        List<Long> zombieJobInstanceIds = Lists.newArrayList();
        ContainerCheckZombieResponse response;
        for (Long jobInstanceId : request.getJobInstanceIdList()) {
            if (!taskMasterPool.contains(jobInstanceId)) {
                zombieJobInstanceIds.add(jobInstanceId);
            }
        }
        response = ContainerCheckZombieResponse.newBuilder()
            .addAllZombieJobInstanceId(zombieJobInstanceIds)
            .build();
        getSender().tell(response, getSelf());
    }
    
}
