package com.alibaba.schedulerx.worker.batch;

import java.util.Map;

import com.alibaba.schedulerx.protocol.Worker.ContainerReportTaskStatusRequest;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;

import com.google.common.collect.Maps;

/**
 * a reqs handler per jobInstance
 * @author yanxun on 2019/1/7.
 */
public enum ContainerStatusReqHandlerPool {
    /**
     * Singleton
     */
    INSTANCE;
    private static final Logger LOGGER = LogFactory.getLogger(ContainerStatusReqHandlerPool.class);
    
    private Map<Long, ContainerStatusReqHandler<ContainerReportTaskStatusRequest>> handlers = Maps.newConcurrentMap();
    
    public void start(long jobInstanceId, ContainerStatusReqHandler<ContainerReportTaskStatusRequest> reqHandler) {
        if (!handlers.containsKey(jobInstanceId)) {
            // only process init phase;
            synchronized (this) {
                if (!handlers.containsKey(jobInstanceId)) {
                    // make sure no other already create mapping during sync blocking time range.
                    handlers.put(jobInstanceId, reqHandler);
                    reqHandler.start();
                }
            }
        }
    }
    
    public void stop(long jobInstanceId) {
        if (handlers.containsKey(jobInstanceId)) {
            handlers.get(jobInstanceId).stop();
            handlers.remove(jobInstanceId);
        }
    }
    
    public boolean contains(long jobInstanceId) {
        return handlers.containsKey(jobInstanceId);
    }
    
    public boolean submitReq(long jobInstanceId, ContainerReportTaskStatusRequest request) {
        boolean success;
        try {
            ContainerStatusReqHandler handler = handlers.get(jobInstanceId);
            if (handler == null) {
                LOGGER.warn("Report task status failed: handler is not exist, JobInstanceId={} maybe has been destroyed, uniqueId:{}_{}_{}", jobInstanceId, request.getJobId(),
                        request.getJobInstanceId(), request.getTaskId());
                return false;
            }
            handler.submitRequest(request);
            success = true;
        } catch (Throwable e) {
            success = false;
            LOGGER.error("JobInstanceId={}, Container buffer status req error, uniqueId:{}_{}_{}", jobInstanceId, request.getJobId(),
                request.getJobInstanceId(), request.getTaskId(), e);
        }
        return success;
    }
    
    public Map<Long, ContainerStatusReqHandler<ContainerReportTaskStatusRequest>> getHandlers() {
        return handlers;
    }
}
