package com.alibaba.schedulerx.worker.container;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicLong;

import com.alibaba.schedulerx.common.util.IdUtil;
import com.alibaba.schedulerx.worker.domain.JobContext;

import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;

/**
 *
 * @author xiaomeng.hxm
 */
public abstract class ContainerPool {

    private Map<String, Container> containerMap = Maps.newConcurrentMap();

    private Map<Long, String> instanceMasterActorPathMap = Maps.newConcurrentMap();

    protected Map<Long, Set<Future>> futureMap = Maps.newConcurrentMap();

    protected Map<String, AtomicLong> groupCountMap = Maps.newConcurrentMap();

    public Map<String, Container> getContainerMap() {
        return containerMap;
    }

    public void setContainerMap(Map<String, Container> containerMap) {
        this.containerMap = containerMap;
    }

    public Container get(String uniqueId) {
        return containerMap.get(uniqueId);
    }

    public void put(String uniqueId, Container container) {
        containerMap.put(uniqueId, container);
        if (container instanceof ThreadContainer) {
            String masterActorPath = ((ThreadContainer) container).getContext().getInstanceMasterActorPath();
            long jobInstanceId = ((ThreadContainer) container).getContext().getJobInstanceId();
            instanceMasterActorPathMap.put(jobInstanceId, masterActorPath);
        }
    }

    public Map<Long, String> getInstanceMasterActorPathMap() {
        return instanceMasterActorPathMap;
    }

    public void remove(String uniqueId) {
        Container container = containerMap.remove(uniqueId);
        long jobInstanceId = IdUtil.parse(uniqueId, IdUtil.IdType.JOB_INSTANCE_ID);
        Set<Future> futureSet = futureMap.get(jobInstanceId);
        if (CollectionUtils.isNotEmpty(futureSet)) {
            if (container instanceof ThreadContainer) {
                Future future = ((ThreadContainer) container).getFuture();
                if (future != null) {
                    futureSet.remove(future);
                }
            }
        }
    }

    public boolean contain(String uniqueId) {
        return containerMap.containsKey(uniqueId);
    }

    /**
     * submit task container
     * @param jobId job id
     * @param jobInstanceId instance identification
     * @param taskId task identification
     * @param container running container
     * @param consumerSize running container service max capacity
     */
    public abstract void submit(long jobId, long jobInstanceId, long taskId, Container container, int consumerSize);

    /**
     * destroy pool by jobInstance
     * @param jobInstanceId instance identification
     */
    public boolean destroyByInstance(long jobInstanceId, boolean mayInterruptIfRunning) {
        this.instanceMasterActorPathMap.remove(jobInstanceId);
        return true;
    }

    public abstract void setContext(JobContext jobContext);

    public abstract JobContext getContext();

    public abstract void removeContext();

    /**
     * 获取指定任务实例锁对象
     * @param jobInstanceId
     * @return
     */
    public abstract AtomicLong getInstanceLock(long jobInstanceId, Long serialNum);

    /**
     * 清除任务实例锁对象
     * @param jobInstanceId
     */
    public abstract void releaseInstanceLock(long jobInstanceId);

    /**
     * 停止容器
     */
    public abstract void shutdown(ShutdownMode mode) throws InterruptedException;

    /**
     * 创建执行线程名
     * @param jobId
     * @param jobInstanceId
     * @param taskId
     * @return
     */
    public abstract String genThreadName(Long jobId, Long jobInstanceId, Long taskId);

    /**
     * 记录任务执行次数
     */
    protected void record(Long jobId, String groupId) {
        AtomicLong count = groupCountMap.get(groupId);
        if (count == null) {
            synchronized (ContainerPool.class) {
                count = groupCountMap.get(groupId);
                if (count == null) {
                    count = new AtomicLong(0);
                    groupCountMap.put(groupId, count);
                }
            }
        }
        count.incrementAndGet();
    }

    /**
     * getCount
     * @param groupId
     * @return
     */
    public long getCount(String groupId){
        AtomicLong count = groupCountMap.get(groupId);
        return count != null?count.get():0;
    }
}
