package com.alibaba.schedulerx.worker.log.appender;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginConfiguration;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.util.Booleans;
import org.apache.logging.log4j.core.util.Throwables;
import org.apache.logging.log4j.status.StatusLogger;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.SlsInfo;
import com.alibaba.schedulerx.common.util.Base64Util;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.container.ContainerFactory;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.aliyun.openservices.aliyun.log.producer.LogProducer;
import com.aliyun.openservices.aliyun.log.producer.Producer;
import com.aliyun.openservices.aliyun.log.producer.ProducerConfig;
import com.aliyun.openservices.aliyun.log.producer.ProjectConfig;
import com.aliyun.openservices.aliyun.log.producer.errors.TimeoutException;
import com.aliyun.openservices.log.common.LogItem;

/**
 * @author xiaomeng.hxm
 *
 */
@Plugin(name = "SchedulerxLog4j2Appender", category = "Core", elementType = "appender", printObject = true)
public class SchedulerxLog4j2Appender extends AbstractAppender {

    private static final String DEFAULT_TIME_FORMAT = "yyyy-MM-dd'T'HH:mm:ssZ";
    private static final String DEFAULT_TIME_ZONE = "UTC";
    
    private String project;
    private String logStore;
    private String endpoint;
    private String accessKeyId;
    private String accessKeySecret;

    private String userAgent = "log4j2";
    private Producer producer;
    private String source;
    private ProducerConfig producerConfig = new ProducerConfig();

    private DateTimeFormatter formatter;
    
    private static final Logger logger = StatusLogger.getLogger();
    private LogCollector logCollector = LogCollectorFactory.get();
    
    protected SchedulerxLog4j2Appender(String name,
                             Filter filter,
                             Layout<? extends Serializable> layout,
                             boolean ignoreExceptions                      
    ) {
        super(name, filter, layout, ignoreExceptions);
    }

    @Override
    public void start() {
        try {
            super.start();
    
            producerConfig.setMaxBlockMs(0);
            producerConfig.setTotalSizeInBytes(20 * 1024 * 1024);
    
            ProjectConfig projectConfig = buildProjectConfig();
            if (projectConfig != null) {
                producer = new LogProducer(producerConfig);
                producer.putProjectConfig(projectConfig);
            }
        } catch (Exception e) {
            logger.error("", e);
        }
    }

    private ProjectConfig buildProjectConfig() {
        Configuration conf = ConfigUtil.getWorkerConfig();
        String logServiceType = conf.getString(CommonConstants.LOG_COLLECTOR_TYPE, CommonConstants.LOG_COLLECTOR_TYPE_DEFAULT);
        if (logServiceType.equalsIgnoreCase("sls")) {
            String logConfig = null;
            String[] logConfigList = conf.getStringArray(CommonConstants.LOG_SERVICE_CONFIG);
            if (logConfigList.length > 1) {
                logConfig = StringUtils.join(logConfigList, ",");
            } else if (logConfigList.length == 1){
                logConfig = logConfigList[0];
            }
            
            if (StringUtils.isNotEmpty(logConfig) && !logConfig.equalsIgnoreCase("null")) {
                SlsInfo slsInfo = JsonUtil.fromJson(logConfig, SlsInfo.class);
                endpoint = slsInfo.getEndpoint();
                project = slsInfo.getProject();
                logStore = slsInfo.getLogstore();
                
                if (conf.getBoolean(CommonConstants.SLS_AKSK_ENCODED, false)) {
                    String encodedAccessKeyId = conf.getString(CommonConstants.SLS_AK);
                    accessKeyId = Base64Util.decode(encodedAccessKeyId);
                    String encodedAccessKeySecret = conf.getString(CommonConstants.SLS_SK);
                    accessKeySecret = Base64Util.decode(encodedAccessKeySecret);
                } else {
                    accessKeyId = conf.getString(CommonConstants.SLS_AK);
                    accessKeySecret = conf.getString(CommonConstants.SLS_SK);
                }
                if (project != null && logStore != null && endpoint != null && accessKeyId != null && accessKeySecret !=null) {
                    LogConfig.INSTANCE.setEnable(true);
                    return new ProjectConfig(project, endpoint, accessKeyId, accessKeySecret);
                }
            }
        }
        
        return null;
    }

    @Override
    public void stop() {
        super.stop();
        if (producer != null) {
            try {
                producer.close();
            } catch (Exception e) {
                logger.error("Failed to close LoghubAppender.", e);
            }
        }
    }

    @Override
    public void append(LogEvent event) {
        JobContext jobContext = ContainerFactory.getContainerPool().getContext();
        //如果不在ThreadContainer的线程中，当前版本不收集
        if (jobContext == null) {
            logger.warn("jobContext is null, threadName={}, stackTrace={}, message={}", 
                event.getThreadName(), event.getSource(), event.getMessage().getFormattedMessage());
            return;
        }
        
        if (producer == null) {
            ProjectConfig projectConfig = buildProjectConfig();
            if (projectConfig != null) {
                producer = new LogProducer(producerConfig);
                producer.putProjectConfig(projectConfig);
            }
        }
        if (producer == null) {
            return;
        }
        
        if (!LogConfig.INSTANCE.isEnable()) {
            return;
        }
        
        List<LogItem> logItems = new ArrayList<LogItem>();
        LogItem item = new LogItem();
        logItems.add(item);
        item.SetTime((int) (event.getTimeMillis() / 1000));
        DateTime dateTime = new DateTime(event.getTimeMillis());
        item.PushBack("executionId", jobContext.getUniqueId());
        item.PushBack("time", dateTime.toString(formatter));
        item.PushBack("level", event.getLevel().toString());
        item.PushBack("thread", event.getThreadName());
        item.PushBack("group", "schedulerx-user");

        StackTraceElement source = event.getSource();
        if (source == null && (!event.isIncludeLocation())) {
            event.setIncludeLocation(true);
            source = event.getSource();
            event.setIncludeLocation(false);
        }

        item.PushBack("location", source == null ? "Unknown(Unknown Source)" : source.toString());

        String message = event.getMessage().getFormattedMessage();
        item.PushBack("message", message);

        String throwable = getThrowableStr(event.getThrown());
        if (throwable != null) {
            item.PushBack("throwable", throwable);
        }

        if (getLayout() != null) {
            item.PushBack("log", new String(getLayout().toByteArray(event)));
        } else {
            item.PushBack("log", message);
        }

        String topic = createTopic(jobContext);
        try {
            producer.send(this.project, this.logStore, topic, this.source, logItems, new SchedulerxLog4j2AppenderCallback(LOGGER,
                    this.project, this.logStore, topic, this.source, logItems));
        } catch (TimeoutException e) {
            //noting to do
            logger.error("send sls log timeout, threadName=" + event.getThreadName()
                + ", stackTrace=" + event.getSource());
        } catch (Exception e) {
            logger.error(
                    "Failed to send log, project=" + project
                            + ", logStore=" + logStore
                            + ", topic=" + topic
                            + ", source=" + source
                            + ", logItem=" + logItems, e);
        }
    }

    /**
     * createTopic
     * @param jobContext
     * @return
     */
    private String createTopic(JobContext jobContext) {
        return "group-" + jobContext.getAppGroupId();
    }

    private String getThrowableStr(Throwable throwable) {
        if (throwable == null) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        boolean isFirst = true;
        for (String s : Throwables.toStringList(throwable)) {
            if (isFirst) {
                isFirst = false;
            } else {
                sb.append(System.getProperty("line.separator"));
            }
            sb.append(s);
        }
        return sb.toString();
    }

    @PluginFactory
    public static SchedulerxLog4j2Appender createAppender(
            @PluginAttribute("name") final String name,
            @PluginElement("Filter") final Filter filter,
            @PluginElement("Layout") Layout<? extends Serializable> layout,
            @PluginConfiguration final Configuration config,
            @PluginAttribute("ignoreExceptions") final String ignore,
            @PluginAttribute("project") final String project,
            @PluginAttribute("logStore") final String logStore,
            @PluginAttribute("endpoint") final String endpoint,
            @PluginAttribute("accessKeyId") final String accessKeyId,
            @PluginAttribute("accessKeySecret") final String accessKeySecret,
            @PluginAttribute("stsToken") final String stsToken,

            @PluginAttribute("totalSizeInBytes") final String  totalSizeInBytes,
            @PluginAttribute("maxBlockMs") final String  maxBlockMs,
            @PluginAttribute("ioThreadCount") final String  ioThreadCount,
            @PluginAttribute("batchSizeThresholdInBytes") final String  batchSizeThresholdInBytes,
            @PluginAttribute("batchCountThreshold") final String  batchCountThreshold,
            @PluginAttribute("lingerMs") final String  lingerMs,
            @PluginAttribute("retries") final String  retries,
            @PluginAttribute("baseRetryBackoffMs") final String  baseRetryBackoffMs,
            @PluginAttribute("maxRetryBackoffMs") final String maxRetryBackoffMs,

            @PluginAttribute("topic") final String topic,
            @PluginAttribute("source") final String source,
            @PluginAttribute("timeFormat") final String timeFormat,
            @PluginAttribute("timeZone") final String timeZone,
            @PluginAttribute("mdcFields") final String mdcFields) {

        Boolean ignoreExceptions = Booleans.parseBoolean(ignore, true);

        int maxBlockMsInt = parseStrToInt(maxBlockMs, 0);

        int baseRetryBackoffMsInt = parseStrToInt(baseRetryBackoffMs, 100);

        int maxRetryBackoffMsInt = parseStrToInt(maxRetryBackoffMs, 100);

        int lingerMsInt = parseStrToInt(lingerMs, 3000);

        int batchCountThresholdInt = parseStrToInt(batchCountThreshold, 4096);

        int batchSizeThresholdInBytesInt = parseStrToInt(batchSizeThresholdInBytes, 5 * 1024 * 1024);

        int totalSizeInBytesInt = parseStrToInt(totalSizeInBytes, 10485760);

        int retriesInt = parseStrToInt(retries, 3);

        int ioThreadCountInt = parseStrToInt(ioThreadCount, 8);

        String pattern = isStrEmpty(timeFormat) ? DEFAULT_TIME_FORMAT : timeFormat;
        String timeZoneInfo = isStrEmpty(timeZone) ? DEFAULT_TIME_ZONE : timeZone;
        DateTimeFormatter formatter = DateTimeFormat.forPattern(pattern).withZone(DateTimeZone.forID(timeZoneInfo));

        return new SchedulerxLog4j2Appender(name, filter, layout, ignoreExceptions);
    }

    static boolean isStrEmpty(String str) {
        return str == null || str.length() == 0;
    }

    static int parseStrToInt(String str, final int defaultVal) {
        if (!isStrEmpty(str)) {
            try {
                return Integer.valueOf(str);
            } catch (NumberFormatException e) {
                return defaultVal;
            }
        } else {
            return defaultVal;
        }
    }

    static void checkCondition(Boolean condition, String errorMsg) {
        if (!condition) {
            throw new IllegalArgumentException(errorMsg);
        }
    }

    public String getProject() {
        return project;
    }

    public void setProject(String project) {
        this.project = project;
    }

    public String getEndpoint() {
        return endpoint;
    }

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public String getAccessKeyId() {
        return accessKeyId;
    }

    public void setAccessKeyId(String accessKeyId) {
        this.accessKeyId = accessKeyId;
    }

    public String getAccessKeySecret() {
        return accessKeySecret;
    }

    public void setAccessKeySecret(String accessKeySecret) {
        this.accessKeySecret = accessKeySecret;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public void setUserAgent(String userAgent) {
        this.userAgent = userAgent;
    }

    public String getLogStore() {
        return logStore;
    }

    public void setLogStore(String logStore) {
        this.logStore = logStore;
    }

    public int getTotalSizeInBytes() {
        return producerConfig.getTotalSizeInBytes();
    }

    public void setTotalSizeInBytes(int totalSizeInBytes) {
        producerConfig.setTotalSizeInBytes(totalSizeInBytes);
    }

    public long getMaxBlockMs() {
        return producerConfig.getMaxBlockMs();
    }

    public void setMaxBlockMs(long maxBlockMs) {
        producerConfig.setMaxBlockMs(maxBlockMs);
    }

    public int getIoThreadCount() {
        return producerConfig.getIoThreadCount();
    }

    public void setIoThreadCount(int ioThreadCount) {
        producerConfig.setIoThreadCount(ioThreadCount);
    }

    public int getBatchSizeThresholdInBytes() {
        return producerConfig.getBatchSizeThresholdInBytes();
    }

    public void setBatchSizeThresholdInBytes(int batchSizeThresholdInBytes) {
        producerConfig.setBatchSizeThresholdInBytes(batchSizeThresholdInBytes);
    }

    public int getBatchCountThreshold() {
        return producerConfig.getBatchCountThreshold();
    }

    public void setBatchCountThreshold(int batchCountThreshold) {
        producerConfig.setBatchCountThreshold(batchCountThreshold);
    }

    public int getLingerMs() {
        return producerConfig.getLingerMs();
    }

    public void setLingerMs(int lingerMs) {
        producerConfig.setLingerMs(lingerMs);
    }

    public int getRetries() {
        return producerConfig.getRetries();
    }

    public void setRetries(int retries) {
        producerConfig.setRetries(retries);
    }

    public int getMaxReservedAttempts() {
        return producerConfig.getMaxReservedAttempts();
    }

    public void setMaxReservedAttempts(int maxReservedAttempts) {
        producerConfig.setMaxReservedAttempts(maxReservedAttempts);
    }

    public long getBaseRetryBackoffMs() {
        return producerConfig.getBaseRetryBackoffMs();
    }

    public void setBaseRetryBackoffMs(long baseRetryBackoffMs) {
        producerConfig.setBaseRetryBackoffMs(baseRetryBackoffMs);
    }

    public long getMaxRetryBackoffMs() {
        return producerConfig.getMaxRetryBackoffMs();
    }

    public void setMaxRetryBackoffMs(long maxRetryBackoffMs) {
        producerConfig.setMaxRetryBackoffMs(maxRetryBackoffMs);
    }

    public String getSource() {
        return source;
    }

    public void setSource(String source) {
        this.source = source;
    }

}