package com.alibaba.schedulerx.worker.log.appender;

import java.time.Instant;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.SlsInfo;
import com.alibaba.schedulerx.common.util.Base64Util;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.container.ContainerFactory;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.aliyun.openservices.aliyun.log.producer.LogProducer;
import com.aliyun.openservices.aliyun.log.producer.Producer;
import com.aliyun.openservices.aliyun.log.producer.ProducerConfig;
import com.aliyun.openservices.aliyun.log.producer.ProjectConfig;
import com.aliyun.openservices.aliyun.log.producer.errors.ProducerException;
import com.aliyun.openservices.aliyun.log.producer.errors.TimeoutException;
import com.aliyun.openservices.log.common.LogItem;

import ch.qos.logback.classic.spi.IThrowableProxy;
import ch.qos.logback.classic.spi.LoggingEvent;
import ch.qos.logback.classic.spi.StackTraceElementProxy;
import ch.qos.logback.classic.spi.ThrowableProxyUtil;
import ch.qos.logback.core.CoreConstants;
import ch.qos.logback.core.UnsynchronizedAppenderBase;
import ch.qos.logback.core.encoder.Encoder;

/**
 * @author xiaomeng.hxm
 *
 */
public class SchedulerxLogbackAppender<E> extends UnsynchronizedAppenderBase<E> {

    private String project;

    private String endpoint;

    private String accessKeyId;

    private String accessKeySecret;

    private String userAgent = "logback";

    protected Encoder<E> encoder;

    protected ProducerConfig producerConfig = new ProducerConfig();

    protected Producer producer;

    protected String logStore; //
    protected String topic = ""; //
    protected String source = ""; //

    protected String timeZone = "UTC";
    protected String timeFormat = "yyyy-MM-dd'T'HH:mmZ";
    protected DateTimeFormatter formatter;
    protected java.time.format.DateTimeFormatter formatter1;
    
    private LogCollector logCollector = LogCollectorFactory.get();
    
    @Override
    public void start() {
        try {
            doStart();
        } catch (Exception e) {
            addError("Failed to start SchedulerxLogbackAppender.", e);
        }
    }

    private void doStart() {
        try {
            formatter = DateTimeFormat.forPattern(timeFormat).withZone(DateTimeZone.forID(timeZone));
        } catch (Exception e){
            formatter1 = java.time.format.DateTimeFormatter.ofPattern(timeFormat).withZone(ZoneId.of(timeZone));
        }
        
        super.start();
        
        producerConfig.setMaxBlockMs(0);
        producerConfig.setTotalSizeInBytes(20 * 1024 * 1024);
        ProjectConfig projectConfig = buildProjectConfig();
        if (projectConfig != null) {
            producer = new LogProducer(producerConfig);
            producer.putProjectConfig(projectConfig);
        }
    }

    private ProjectConfig buildProjectConfig() {
        Configuration conf = ConfigUtil.getWorkerConfig();
        String logServiceType = conf.getString(CommonConstants.LOG_COLLECTOR_TYPE, CommonConstants.LOG_COLLECTOR_TYPE_DEFAULT);
        if (logServiceType.equalsIgnoreCase("sls")) {
            String logConfig = null;
            String[] logConfigList = conf.getStringArray(CommonConstants.LOG_SERVICE_CONFIG);
            if (logConfigList.length > 1) {
                logConfig = StringUtils.join(logConfigList, ",");
            } else if (logConfigList.length == 1){
                logConfig = logConfigList[0];
            }
            if (StringUtils.isNotEmpty(logConfig) && !logConfig.equalsIgnoreCase("null")) {
                SlsInfo slsInfo = JsonUtil.fromJson(logConfig, SlsInfo.class);
                endpoint = slsInfo.getEndpoint();
                project = slsInfo.getProject();
                logStore = slsInfo.getLogstore();
                
                if (conf.getBoolean(CommonConstants.SLS_AKSK_ENCODED, false)) {
                    String encodedAccessKeyId = conf.getString(CommonConstants.SLS_AK);
                    accessKeyId = Base64Util.decode(encodedAccessKeyId);
                    String encodedAccessKeySecret = conf.getString(CommonConstants.SLS_SK);
                    accessKeySecret = Base64Util.decode(encodedAccessKeySecret);
                } else {
                    accessKeyId = conf.getString(CommonConstants.SLS_AK);
                    accessKeySecret = conf.getString(CommonConstants.SLS_SK);
                }
                if (project != null && logStore != null && endpoint != null && accessKeyId != null && accessKeySecret !=null) {
                    LogConfig.INSTANCE.setEnable(true);
                    return new ProjectConfig(project, endpoint, accessKeyId, accessKeySecret);
                }
            }
        }
        
        return null;
    }

    @Override
    public void stop() {
        try {
            doStop();
        } catch (Exception e) {
            addError("Failed to stop SchedulerxLogbackAppender.", e);
        }
    }

    private void doStop() throws InterruptedException, ProducerException {
        if (!isStarted()) {
            return;
        }

        super.stop();
        if(producer!=null) {
            producer.close();
        }
    }

    @Override
    public void append(E eventObject) {
        try {
            appendEvent(eventObject);
        } catch (Exception e) {
            addError("Failed to append event.", e);
        }
    }

    private void appendEvent(E eventObject) {
        //init Event Object
        if (!(eventObject instanceof LoggingEvent)) {
            return;
        }
        
        LoggingEvent event = (LoggingEvent) eventObject;
        
        JobContext jobContext = ContainerFactory.getContainerPool().getContext();
        //如果不在ThreadContainer的线程中，当前版本不收集
        if (jobContext == null) {
            addWarn("jobContext is null, threadName=" + event.getThreadName()
                + ", stackTrace=" + event.getCallerData() + ", message=" + event.getFormattedMessage());
            return;
        }
        
        if (producer == null) {
            ProjectConfig projectConfig = buildProjectConfig();
            if (projectConfig != null) {
                producer = new LogProducer(producerConfig);
                producer.putProjectConfig(projectConfig);
            }
        }
        if (producer == null) {
            return;
        }
        
        if (!LogConfig.INSTANCE.isEnable()) {
            return;
        }
        
        List<LogItem> logItems = new ArrayList<LogItem>();
        LogItem item = new LogItem();
        logItems.add(item);
        item.SetTime((int) (event.getTimeStamp() / 1000));
        item.PushBack("executionId", jobContext.getUniqueId());
        if (formatter != null) {
            DateTime dateTime = new DateTime(event.getTimeStamp());
            item.PushBack("time", dateTime.toString(formatter));
        } else {
            Instant instant = Instant.ofEpochMilli(event.getTimeStamp());
            item.PushBack("time", formatter1.format(instant));
        }

        item.PushBack("level", event.getLevel().toString());
        item.PushBack("thread", event.getThreadName());
        item.PushBack("group", "schedulerx-user");

        StackTraceElement[] caller = event.getCallerData();
        if (caller != null && caller.length > 0) {
            item.PushBack("location", caller[0].toString());
        }

        String message = event.getFormattedMessage();
        item.PushBack("message", message);

        IThrowableProxy iThrowableProxy = event.getThrowableProxy();
        if (iThrowableProxy != null) {
            String throwable = getExceptionInfo(iThrowableProxy);
            throwable += fullDump(event.getThrowableProxy().getStackTraceElementProxyArray());
            item.PushBack("throwable", throwable);
        }

        if (this.encoder != null) {
            item.PushBack("log", new String(this.encoder.encode(eventObject)));
        } else {
            item.PushBack("log", message);
        }
        
        try {
            String topic = createTopic(jobContext);
            producer.send(this.project, logStore, topic, source, logItems, new SchedulerxLogbackAppenderCallBack<E>(this,
                    this.project, logStore, topic, source, logItems));
        } catch (TimeoutException e) {
            addError("send sls log timeout, threadName=" + event.getThreadName()
            + ", stackTrace=" + event.getCallerData());
        } catch (Exception e) {
            addError("Failed to send log.", e);
        }
    }

    /**
     * createTopic
     * @param jobContext
     * @return
     */
    private String createTopic(JobContext jobContext) {
        return "group-" + jobContext.getAppGroupId();
    }

    public String getTimeFormat() {
        return timeFormat;
    }

    public void setTimeFormat(String timeFormat) {
        this.timeFormat = timeFormat;
    }

    private String getExceptionInfo(IThrowableProxy iThrowableProxy) {
        String s = iThrowableProxy.getClassName();
        String message = iThrowableProxy.getMessage();
        return (message != null) ? (s + ": " + message) : s;
    }

    private String fullDump(StackTraceElementProxy[] stackTraceElementProxyArray) {
        StringBuilder builder = new StringBuilder();
        for (StackTraceElementProxy step : stackTraceElementProxyArray) {
            builder.append(CoreConstants.LINE_SEPARATOR);
            String string = step.toString();
            builder.append(CoreConstants.TAB).append(string);
            ThrowableProxyUtil.subjoinPackagingData(builder, step);
        }
        return builder.toString();
    }

    public String getLogStore() {
        return logStore;
    }

    public void setLogStore(String logStore) {
        this.logStore = logStore;
    }

    public String getTopic() {
        return topic;
    }

    public void setTopic(String topic) {
        this.topic = topic;
    }

    public String getSource() {
        return source;
    }

    public void setSource(String source) {
        this.source = source;
    }

    public String getTimeZone() {
        return timeZone;
    }

    public void setTimeZone(String timeZone) {
        this.timeZone = timeZone;
    }

    // **** ==- ProjectConfig -== **********************
    public String getProject() {
        return project;
    }

    public void setProject(String project) {
        this.project = project;
    }

    public String getEndpoint() {
        return endpoint;
    }

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public String getAccessKeyId() {
        return accessKeyId;
    }

    public void setAccessKeyId(String accessKeyId) {
        this.accessKeyId = accessKeyId;
    }

    public String getAccessKeySecret() {
        return accessKeySecret;
    }

    public void setAccessKeySecret(String accessKeySecret) {
        this.accessKeySecret = accessKeySecret;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public void setUserAgent(String userAgent) {
        this.userAgent = userAgent;
    }

    public int getTotalSizeInBytes() {
        return producerConfig.getTotalSizeInBytes();
    }

    public void setTotalSizeInBytes(int totalSizeInBytes) {
        producerConfig.setTotalSizeInBytes(totalSizeInBytes);
    }

    public long getMaxBlockMs() {
        return producerConfig.getMaxBlockMs();
    }

    public void setMaxBlockMs(long maxBlockMs) {
        producerConfig.setMaxBlockMs(maxBlockMs);
    }

    public int getIoThreadCount() {
        return producerConfig.getIoThreadCount();
    }

    public void setIoThreadCount(int ioThreadCount) {
        producerConfig.setIoThreadCount(ioThreadCount);
    }

    public int getBatchSizeThresholdInBytes() {
        return producerConfig.getBatchSizeThresholdInBytes();
    }

    public void setBatchSizeThresholdInBytes(int batchSizeThresholdInBytes) {
        producerConfig.setBatchSizeThresholdInBytes(batchSizeThresholdInBytes);
    }

    public int getBatchCountThreshold() {
        return producerConfig.getBatchCountThreshold();
    }

    public void setBatchCountThreshold(int batchCountThreshold) {
        producerConfig.setBatchCountThreshold(batchCountThreshold);
    }

    public int getLingerMs() {
        return producerConfig.getLingerMs();
    }

    public void setLingerMs(int lingerMs) {
        producerConfig.setLingerMs(lingerMs);
    }

    public int getRetries() {
        return producerConfig.getRetries();
    }

    public void setRetries(int retries) {
        producerConfig.setRetries(retries);
    }

    public int getMaxReservedAttempts() {
        return producerConfig.getMaxReservedAttempts();
    }

    public void setMaxReservedAttempts(int maxReservedAttempts) {
        producerConfig.setMaxReservedAttempts(maxReservedAttempts);
    }

    public long getBaseRetryBackoffMs() {
        return producerConfig.getBaseRetryBackoffMs();
    }

    public void setBaseRetryBackoffMs(long baseRetryBackoffMs) {
        producerConfig.setBaseRetryBackoffMs(baseRetryBackoffMs);
    }

    public long getMaxRetryBackoffMs() {
        return producerConfig.getMaxRetryBackoffMs();
    }

    public void setMaxRetryBackoffMs(long maxRetryBackoffMs) {
        producerConfig.setMaxRetryBackoffMs(maxRetryBackoffMs);
    }

    public Encoder<E> getEncoder() {
        return encoder;
    }

    public void setEncoder(Encoder<E> encoder) {
        this.encoder = encoder;
    }

}

