package com.alibaba.schedulerx.worker.logcollector;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;

import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.SlsInfo;
import com.alibaba.schedulerx.common.domain.StreamType;
import com.alibaba.schedulerx.common.util.Base64Util;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.log.appender.LogConfig;
import com.aliyun.openservices.aliyun.log.producer.Callback;
import com.aliyun.openservices.aliyun.log.producer.LogProducer;
import com.aliyun.openservices.aliyun.log.producer.Producer;
import com.aliyun.openservices.aliyun.log.producer.ProducerConfig;
import com.aliyun.openservices.aliyun.log.producer.ProjectConfig;
import com.aliyun.openservices.aliyun.log.producer.Result;
import com.aliyun.openservices.log.common.LogItem;

/**
 *
 * @author xiaomeng.hxm
 */
public class SlsLogCollector extends LogCollector {
    private static final Logger LOGGER = LogFactory.getLogger(SlsLogCollector.class);
    private String project;
    private String logStore;
    private String endpoint;
    private String accessKeyId;
    private String accessKeySecret;
    private Producer producer;
    private ProducerConfig producerConfig;

    public SlsLogCollector() {
        producerConfig = new ProducerConfig();
        producerConfig.setMaxBlockMs(0);
        producerConfig.setTotalSizeInBytes(20 * 1024 * 1024);
        ProjectConfig projectConfig = buildProjectConfig();
        if (projectConfig != null) {
            producer = new LogProducer(producerConfig);
            producer.putProjectConfig(projectConfig);
        }
    }

    @Override
    public void collect(final long appGroupId, final String key, final String line, final Throwable t, StreamType streamType, boolean isEnd) {
        try {
            if(StringUtils.isEmpty(line)){
                return;
            }
            if (producer == null) {
                ProjectConfig projectConfig = buildProjectConfig();
                if (projectConfig != null) {
                    producer = new LogProducer(producerConfig);
                    producer.putProjectConfig(projectConfig);
                }
            }
            if (producer == null) {
                return;
            }

            if (!LogConfig.INSTANCE.isEnable()) {
                return;
            }
            
            producer.send(project, logStore, createTopic(appGroupId), SchedulerxWorker.WORKER_ADDR, generateLogItems(key, line, t, streamType), new Callback() {
                @Override
                public void onCompletion(Result result) {
                if (result.isSuccessful()) {
                    //TODO 这里做日志计量统计
                    LOGGER.debug("");
                } else {
                    LOGGER.error("send sls log failed, " + result);
                }
                }

            });
        } catch (Exception e) {
            LOGGER.warn("SlS log send failed. message:{}, logContent:{}", e.getMessage(), line);
        }
    }

    /**
     * createTopic
     * @param jobContext
     * @return
     */
    private String createTopic(long appGroupId) {
       return "group-" + appGroupId;
    }
    
    private ProjectConfig buildProjectConfig() {
        Configuration conf = ConfigUtil.getWorkerConfig();
        String logServiceType = conf.getString(CommonConstants.LOG_COLLECTOR_TYPE, CommonConstants.LOG_COLLECTOR_TYPE_DEFAULT);
        if (logServiceType.equalsIgnoreCase("sls")) {
            String logConfig = null;
            String[] logConfigList = conf.getStringArray(CommonConstants.LOG_SERVICE_CONFIG);
            if (logConfigList.length > 1) {
                logConfig = StringUtils.join(logConfigList, ",");
            } else if (logConfigList.length == 1){
                logConfig = logConfigList[0];
            }
            
            if (StringUtils.isNotEmpty(logConfig) && !logConfig.equalsIgnoreCase("null")) {
                SlsInfo slsInfo = JsonUtil.fromJson(logConfig, SlsInfo.class);
                endpoint = slsInfo.getEndpoint();
                project = slsInfo.getProject();
                logStore = slsInfo.getLogstore();
                
                if (conf.getBoolean(CommonConstants.SLS_AKSK_ENCODED, false)) {
                    String encodedAccessKeyId = conf.getString(CommonConstants.SLS_AK);
                    accessKeyId = Base64Util.decode(encodedAccessKeyId);
                    String encodedAccessKeySecret = conf.getString(CommonConstants.SLS_SK);
                    accessKeySecret = Base64Util.decode(encodedAccessKeySecret);
                } else {
                    accessKeyId = conf.getString(CommonConstants.SLS_AK);
                    accessKeySecret = conf.getString(CommonConstants.SLS_SK);
                }
                if (project != null && logStore != null && endpoint != null && accessKeyId != null && accessKeySecret !=null) {
                    LogConfig.INSTANCE.setEnable(true);
                    return new ProjectConfig(project, endpoint, accessKeyId, accessKeySecret);
                }
            }
        }
        
        return null;
    }

    private static List<LogItem> generateLogItems(String key, String line, Throwable t, StreamType streamType) {
        List<LogItem> logItems = new ArrayList<LogItem>();
        LogItem item = new LogItem();
        logItems.add(item);
        item.SetTime((int) (System.currentTimeMillis() / 1000));
        DateTime dateTime = new DateTime(System.currentTimeMillis());
        item.PushBack("executionId", key);
        item.PushBack("time", dateTime.toString(CommonConstants.DATE_TIME_PATTERN));
        item.PushBack("level", StreamType.STD_ERR.equals(streamType) ? "ERROR" : "INFO");
        item.PushBack("message", line);
        if (t != null) {
            item.PushBack("throwable", ExceptionUtil.getTrace(t));
        }
        item.PushBack("thread", Thread.currentThread().getName());
        item.PushBack("group", "schedulerx-framework");
        item.PushBack("log", line);
        return logItems;
      }

}
