package com.alibaba.schedulerx.worker.processor;

import java.io.IOException;
import java.net.SocketTimeoutException;
import java.net.URLEncoder;

import org.apache.commons.codec.Charsets;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.HttpAttribute;
import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.domain.enums.HttpRespParseModeEnum;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.util.HttpClientUtil;

/**
 * @Auther: 唐涛
 * @Date: 2022/11/22 10:16
 * @Description: agent版本的http任务执行器
 */
public class HttpProcessor implements JobProcessorEx {


    private LogCollector logCollector = LogCollectorFactory.get();
    protected static final Logger LOGGER = LogFactory.getLogger(HttpProcessor.class);

    // 默认获取数据超时10s
    private static final int DEFAULT_SOCKET_TIMEOUT = 10;
    // 默认连接超时3s
    private static final int DEFAULT_CONNECT_TIMEOUT = 3;
    // 连接池获取超时5s
    private static final int DEFAULT_POOL_CONNECT_TIMEOUT = 5;
    private String uniqueId = null;
    CloseableHttpClient httpClient = HttpClientUtil.getCloseableHttpClient();

    @Override
    public void preProcess(JobContext context) throws Exception {

    }

    @Override
    public ProcessResult process(final JobContext context) throws Exception {
        uniqueId = context.getUniqueId();
        HttpProcessResult processResult = null;
        CloseableHttpResponse httpResponse = null;
        try {
            HttpAttribute httpAttribute = JsonUtil.fromJson(context.getContent(), HttpAttribute.class);
            //采集url
            logCollector.collect(context.getAppGroupId(), uniqueId, "Http Request URL: " + httpAttribute.getUrl());
            HttpRequestBase request = buildHttpRequest(context, httpAttribute);
            //采集request header
            logCollector.collect(context.getAppGroupId(), uniqueId, "Http Request Headers: " + JsonUtil.toJson(request.getAllHeaders()));
            //采集入参
            logCollector.collect(context.getAppGroupId(), uniqueId, "Http Request Parameters: " + getHttpRequestParam(context));
            httpResponse = httpClient.execute(request);
            processResult = parseResponse(context, httpResponse, httpAttribute);
            if (processResult.getStatus() == InstanceStatus.SUCCESS) {
                logCollector.collect(context.getAppGroupId(), uniqueId, "http callback success");
            } else {
                String msg = "http callback failed: " + processResult.getResult();
                logCollector.collect(context.getAppGroupId(), uniqueId, msg);
            }
        } catch (SocketTimeoutException | ConnectTimeoutException e1) {
            String msg = String.format("http execute timeout, jobId=%s, jobInstanceId=%s", context.getJobId(), context.getJobInstanceId());
            processResult = new HttpProcessResult(InstanceStatus.FAILED, msg);
            LOGGER.error(msg, e1);
            //发生异常时，采集的日志内容要包括堆栈信息
            logCollector.collect(context.getAppGroupId(), uniqueId, msg + ",throwable: " + ExceptionUtil.getTrace(e1));
        } catch (Throwable ex) {
            String msg = String.format("http execute error, jobId=%s, jobInstanceId=%s", context.getJobId(), context.getJobInstanceId());
            processResult = new HttpProcessResult(InstanceStatus.FAILED, msg);
            LOGGER.error(msg, ex);
            logCollector.collect(context.getAppGroupId(), uniqueId, msg + ",throwable: " + ExceptionUtil.getTrace(ex));
        } finally {
            if (httpResponse != null) {
                try {
                    httpResponse.close();
                } catch (IOException ignored) {
                }
            }
        }
        return processResult;
    }

    @Override
    public ProcessResult postProcess(JobContext context) {
        return null;
    }

    @Override
    public void kill(JobContext context) {

    }


    /**
     * 获取http请求参数
     *
     * @param context
     */
    private String getHttpRequestParam(JobContext context) {
        String params = null;
        if (StringUtils.isNotEmpty(context.getInstanceParameters())) {
            params = context.getInstanceParameters();
        } else if (StringUtils.isNotEmpty(context.getJobParameters())) {
            params = context.getJobParameters();
        }
        return params;
    }

    /**
     * 构建request
     *
     * @param context
     * @param httpAttribute
     * @return
     * @throws IOException
     */
    private HttpRequestBase buildHttpRequest(JobContext context, HttpAttribute httpAttribute) throws IOException {
        HttpRequestBase request = null;
        if (httpAttribute.getTimeout() <= 0) {
            httpAttribute.setTimeout(DEFAULT_SOCKET_TIMEOUT);
        }
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(httpAttribute.getTimeout() * 1000)
                .setConnectTimeout(DEFAULT_CONNECT_TIMEOUT * 1000)
                .setConnectionRequestTimeout(DEFAULT_POOL_CONNECT_TIMEOUT * 1000).build();
        if (HttpGet.METHOD_NAME.equalsIgnoreCase(httpAttribute.getMethod())) {
            request = new HttpGet(httpAttribute.getUrl());
        } else if (HttpPost.METHOD_NAME.equalsIgnoreCase(httpAttribute.getMethod())) {
            request = new HttpPost(httpAttribute.getUrl());
            String params = null;
            if (StringUtils.isNotEmpty(context.getInstanceParameters())) {
                params = context.getInstanceParameters();
            } else if (StringUtils.isNotEmpty(context.getJobParameters())) {
                params = context.getJobParameters();
            }
            //根据请求类型组装参数
            if (StringUtils.isNotBlank(params)) {
                //老数据的contentType 为null,默认是application/x-www-form-urlencoded
                if ("application/x-www-form-urlencoded".equalsIgnoreCase(httpAttribute.getContentType()) || StringUtils.isBlank(httpAttribute.getContentType())) {
                    ContentType contentType = ContentType.create("application/x-www-form-urlencoded", Charsets.UTF_8);
                    StringEntity stringEntity = new StringEntity(params, contentType);
                    ((HttpPost) request).setEntity(stringEntity);
                } else if ("application/json".equalsIgnoreCase(httpAttribute.getContentType())) {
                    ContentType contentType = ContentType.create("application/json", Charsets.UTF_8);
                    StringEntity stringEntity = new StringEntity(params, contentType);
                    ((HttpPost) request).setEntity(stringEntity);
                }
            }

        }

        request.setHeader(CommonConstants.HTTP_JOB_COOKIE_HEADER, httpAttribute.getCookie());
//        request.addHeader("Content-type","application/x-www-form-urlencoded; charset=utf-8");
//        request.addHeader("Accept", "application/x-www-form-urlencoded");
        request.setHeader(CommonConstants.JOB_ID_HEADER, String.valueOf(context.getJobId()));
        request.setHeader(CommonConstants.GROUP_ID_HEADER, String.valueOf(context.getGroupId()));
        request.setHeader(CommonConstants.JOB_NAME_HEADER, URLEncoder.encode(context.getJobName(), Charsets.UTF_8.name()));
        request.addHeader(CommonConstants.SCHEDULE_TIMESTAMP_HEADER, String.valueOf(context.getScheduleTime().getMillis()));
        request.addHeader(CommonConstants.DATA_TIMESTAMP_HEADER, String.valueOf(context.getDataTime().getMillis()));
        request.addHeader(CommonConstants.USER_HEADER, URLEncoder.encode(context.getUser(), Charsets.UTF_8.name()));
        request.addHeader(CommonConstants.MAX_ATTEMPT_HEADER, String.valueOf(context.getMaxAttempt()));
        request.addHeader(CommonConstants.ATTEMPT_HEADER, String.valueOf(context.getAttempt()));
        //设置实例id
        request.addHeader(CommonConstants.JOB_INSTANCE_ID, String.valueOf(context.getJobInstanceId()));

//        request.addHeader(CommonConstants.JOB_PARAMETERS_HEADER, context.getJobParameters());
//        request.addHeader(CommonConstants.INSTANCE_PARAMETERS_HEADER, context.getInstanceParameters());
        request.setConfig(requestConfig);

        return request;
    }


    /**
     * 解析应答结果
     *
     * @param jobInstanceId
     * @param jobId
     * @param httpResponse
     * @param httpAttribute
     * @return
     * @throws IOException
     */
    private HttpProcessResult parseResponse(JobContext context, HttpResponse httpResponse, HttpAttribute httpAttribute) throws IOException {
        //result
        HttpProcessResult result = null;

        // 判断解析模式
        Integer respParseMode = httpAttribute.getRespParseMode();
        if (respParseMode == null) {
            return new HttpProcessResult(InstanceStatus.FAILED, String.format("invalid response parse mode=%s", respParseMode));
        }
        if (httpResponse == null || httpResponse.getStatusLine() == null) {
            return new HttpProcessResult(InstanceStatus.FAILED, "invalid response info");
        }
        // http响应码
        int responseCode = httpResponse.getStatusLine().getStatusCode();
        HttpEntity entity = httpResponse.getEntity();
        //打印返回内容
        String content = null;
        if (entity != null) {
            content = EntityUtils.toString(entity);
            logCollector.collect(context.getAppGroupId(), uniqueId, "Http response content: " + content);
        } else {
            logCollector.collect(context.getAppGroupId(), uniqueId, "Http response content is null");
        }
        // 如果解析模式是自定义JSON
        if (respParseMode.equals(HttpRespParseModeEnum.CUSTOMIZE_JSON.getValue())) {
            if (responseCode == 200) {
                if (entity != null) {
                    try {
                        JSONObject jsonObject = JSON.parseObject(content);
                        String respVal = jsonObject.getString(httpAttribute.getRespKey());
                        if (!httpAttribute.getRespValue().equals(respVal)) {
                            // 返回值不符合预期
                            String msg = String.format("The returned value is different from the expected value: %s", jsonObject.toJSONString());
                            result = new HttpProcessResult(InstanceStatus.FAILED, msg);
                        } else {
                            // 返回结果界面展示
                            result = new HttpProcessResult(InstanceStatus.SUCCESS, jsonObject.toJSONString());
                        }
                    } catch (JSONException e) {
                        // 请求返回不符合json格式
                        String msg = String.format("Json parse failed, jobId=%s, jobInstanceId=%s", context.getJobId(), context.getJobInstanceId());
                        LOGGER.error(msg, e);
                        result = new HttpProcessResult(InstanceStatus.FAILED, msg);
                    }
                } else {
                    result = new HttpProcessResult(InstanceStatus.FAILED, "httpResponse entity is null");
                }
            } else {
                // 非200
                String msg = String.format("http response code is not 200, response code=%s, uri=%s, jobId=%s, jobInstanceId=%s", 
                    responseCode, httpAttribute.getUrl(), context.getJobId(), context.getJobInstanceId());
                LOGGER.error(msg);
                result = new HttpProcessResult(InstanceStatus.FAILED, msg);
            }
        } else if (respParseMode.equals(HttpRespParseModeEnum.RESPONSE_CODE.getValue())) {
            if (!httpAttribute.getRespCode().equals(responseCode)) {
                String msg = String.format("http responseCode is different from the expected value,expected value=%s, "
                    + "actual value=%s", httpAttribute.getRespCode(), responseCode);
                result = new HttpProcessResult(InstanceStatus.FAILED, msg);
            } else {
                result = new HttpProcessResult(InstanceStatus.SUCCESS, content);
            }
        } else if (respParseMode.equals(HttpRespParseModeEnum.CUSTOMIZE_STRING.getValue())) {
            if (responseCode == 200) {
                if (entity != null) {
                    if (!StringUtils.equals(content, httpAttribute.getRespStr())) {
                        String msg = String.format("http returned value different from the expected value, "
                            + "expected value=%s, actual value=%s", httpAttribute.getRespStr(), content);
                        result = new HttpProcessResult(InstanceStatus.FAILED, msg);
                    } else {
                        result = new HttpProcessResult(InstanceStatus.SUCCESS, content);
                    }
                } else {
                    String msg = String.format("http returned value different from the expected value, "
                        + "expected value=%s, actual value=%s", httpAttribute.getRespStr(), content);
                    result = new HttpProcessResult(InstanceStatus.FAILED, msg);
                }
            } else {
                // 非200
                String msg = String.format("http response code is not 200,response code=%s, uri=%s, jobId=%s, "
                    + "jobInstanceId=%s", responseCode, httpAttribute.getUrl(), context.getJobId(), context.getJobInstanceId());
                LOGGER.error(msg);
                result = new HttpProcessResult(InstanceStatus.FAILED, msg);
            }

        } else {
            result = new HttpProcessResult(InstanceStatus.FAILED, String.format("invalid response parse mode=%s", respParseMode));
        }

        return result;
    }

}


