package com.alibaba.schedulerx.worker.processor;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;

import org.apache.commons.lang.StringUtils;
import org.springframework.util.PropertyPlaceholderHelper;

import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.ScriptJobXAttrs;
import com.alibaba.schedulerx.common.domain.StreamType;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.util.ShellUtil;


/**
 *
 * @author xiaomeng.hxm
 */
public class ShellProcessor implements JobProcessorEx {

    private Process shellProcess = null;
    private LogCollector logCollector = LogCollectorFactory.get();
    protected static final Logger LOGGER = LogFactory.getLogger(ShellProcessor.class);
    private long appGroupId;
    private String uniqueId = null;
    private String lastLine = null;

    @SuppressWarnings("finally")
    @Override
    public ProcessResult process(JobContext context) {
        ProcessResult result = new ProcessResult(false);
        try {
            ProcessBuilder processBuilder = ShellUtil.createProcessBuilder(getContent(context));
            if (redirectStream()) {
                processBuilder.redirectErrorStream(true);
            }
            shellProcess = processBuilder.start();

            CountDownLatch countDownLatch = null;
            if (redirectStream()) {
                countDownLatch = new CountDownLatch(1);
                Thread stderrStreamProcessor = new ShellStreamProcessor(this, shellProcess.getInputStream(),
                        StreamType.STD_ERR, countDownLatch);
                stderrStreamProcessor.start();
            } else {
                countDownLatch = new CountDownLatch(2);
                Thread stdoutStreamProcessor = new ShellStreamProcessor(this, shellProcess.getInputStream(),
                        StreamType.STD_OUT, countDownLatch);
                stdoutStreamProcessor.start();

                Thread stderrStreamProcessor = new ShellStreamProcessor(this, shellProcess.getErrorStream(),
                        StreamType.STD_ERR, countDownLatch);
                stderrStreamProcessor.start();
            }

            countDownLatch.await();
            if (shellProcess.waitFor() == 0) {
                result.setStatus(true);
            } else if (lastLine != null) {
                if (lastLine.getBytes().length > CommonConstants.INSTANCE_RESULT_SIZE_MAX) {
                    byte[] tmp = new byte[CommonConstants.INSTANCE_RESULT_SIZE_MAX];
                    System.arraycopy(lastLine.getBytes(), 0, tmp, 0, CommonConstants.INSTANCE_RESULT_SIZE_MAX);
                    result.setResult(new String(tmp));
                } else {
                    result.setResult(lastLine);
                }
            }
        } catch (Throwable e) {
            LOGGER.error("", e);
            logCollector.collect(context.getAppGroupId(), uniqueId, "script process errors", e);
            result.setResult(ExceptionUtil.getMessage(e));
        } finally {
            return result;
        }
    }

    protected String[] getContent(JobContext context) {
        String[] parameters;
        if (StringUtils.isNotEmpty(context.getShardingParameter())) {
            parameters = new String[2];
            parameters[0] = String.valueOf(context.getShardingId());
            parameters[1] = context.getShardingParameter();
        } else if (StringUtils.isNotEmpty(context.getInstanceParameters())) {
            parameters = context.getInstanceParameters().trim().split(" ");
        } else {
            parameters = context.getJobParameters().trim().split(" ");
        }
        String[] contents = new String[3 + parameters.length];
        ScriptJobXAttrs xAttrs = JsonUtil.fromJson(context.getXAttrs(), ScriptJobXAttrs.class);
        if (StringUtils.isNotBlank(xAttrs.getCommand())) {
            contents[0] = xAttrs.getCommand();
        } else {
            contents[0] = "/bin/sh";
        }
        contents[1] = "-c";
        contents[2] = parseContent(context);
        for (int i = 0; i < parameters.length; i++) {
            contents[3+i] = parameters[i];
        }
        return contents;
    }

    /**
     * 解析Content
     */
    protected String parseContent(JobContext context) {
        String content = context.getContent();
        try {
            if (content != null) {
                PropertyPlaceholderHelper helper = new PropertyPlaceholderHelper("#{", "}",
                        null, true);
                Properties properties = new Properties();
                properties.setProperty("schedule.timestamp", Long.toString(context.getScheduleTime().getMillis()));
                properties.setProperty("data.timestamp", Long.toString(context.getScheduleTime().getMillis()));
                content = helper.replacePlaceholders(content, properties);
            }
        }catch (Exception e) {
            LOGGER.warn("ParseContent failed.", e);
        }
        return content;
    }

    protected void processStdOutputStream(InputStream inputStream) {
        String line = null;
        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            while ((line = br.readLine()) != null) {
                logCollector.collect(appGroupId, uniqueId, line);
                lastLine = line;
            }
        } catch (Throwable e) {
            LOGGER.error("error ShellJobProcessor stdout stream", e);
            logCollector.collect(appGroupId, uniqueId, "error process stdout stream", e);
        } finally {
            logCollector.collect(appGroupId, uniqueId, line, StreamType.STD_OUT, true);
        }
    }

    protected void processStdErrorStream(InputStream inputStream) {
        String line = null;
        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            while ((line = br.readLine()) != null) {
                logCollector.collect(appGroupId, uniqueId, line);
                lastLine = line;
            }
        } catch (Throwable e) {
            LOGGER.error("error ShellJobProcessor stderr stream", e);
            logCollector.collect(appGroupId, uniqueId, "error process stderr stream", e);
        } finally {
            logCollector.collect(appGroupId, uniqueId, line, StreamType.STD_OUT, true);
        }
    }

    protected boolean redirectStream() {
        return true;
    }

    @Override
    public ProcessResult postProcess(JobContext context) {
        return null;
    }

    @Override
    public void kill(JobContext context) {
        try {
            long pid = ShellUtil.getPidOfProcess(shellProcess);
            if (pid > 0) {
                ShellUtil.killProcess(pid);
            }
        } catch (Throwable e) {
            LOGGER.error("kill shell job jobInstanceId={} failed, {}", context.getJobInstanceId(), e);
        }

        try {
            if (shellProcess != null) {
                shellProcess.destroy();
            }
        }catch (Throwable th) {
            LOGGER.error("kill shell job jobInstanceId={} failed, {}", context.getJobInstanceId(), th);
        }
    }

    @Override
    public void preProcess(JobContext context) throws Exception {
        uniqueId = context.getUniqueId();
        appGroupId = context.getAppGroupId();
    }

}
