package com.alibaba.schedulerx.worker.processor.springscheduling;

import com.alibaba.schedulerx.scheduling.annotation.SchedulerX;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.core.MethodIntrospector;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.lang.Nullable;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.scheduling.annotation.Schedules;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Bean post-processor that registers methods annotated with @{@link SchedulerX}
 * @author yaohui
 * @create 2022/10/20 下午12:19
 **/
public class SchedulerxAnnotationBeanPostProcessor implements BeanPostProcessor, BeanFactoryAware {

    private final Set<Class<?>> nonAnnotatedClasses = Collections.newSetFromMap(new ConcurrentHashMap(64));

    @Nullable
    private BeanFactory beanFactory;

    @Autowired
    private SchedulerxSchedulingConfigurer schedulerxSchedulingConfigurer;

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        Class<?> targetClass = AopProxyUtils.ultimateTargetClass(bean);
        if (!this.nonAnnotatedClasses.contains(targetClass)) {
            Set<Method> methods = MethodIntrospector.selectMethods(targetClass, new ReflectionUtils.MethodFilter(){
                @Override
                public boolean matches(Method method) {
                    Set<SchedulerX> scheduledMethods = AnnotatedElementUtils.getAllMergedAnnotations(
                            method, SchedulerX.class);
                    return !CollectionUtils.isEmpty(scheduledMethods);
                }
            });
            if (methods != null && methods.size() > 0) {
                for (Method method : methods) {
                    Set<Scheduled> scheduledMethods = AnnotatedElementUtils.getMergedRepeatableAnnotations(
                            method, Scheduled.class, Schedules.class);
                    if (scheduledMethods != null && scheduledMethods.size() > 0) {
                        continue;
                    }
                    schedulerxSchedulingConfigurer.register(bean, method, true);
                }
            }
        }
        return bean;
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public void setBeanFactory(BeanFactory beanFactory) {
        this.beanFactory = beanFactory;
    }
}
