package com.alibaba.schedulerx.worker.processor.springscheduling;

import com.alibaba.schedulerx.common.domain.Pair;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.scheduling.config.Task;
import org.springframework.scheduling.support.ScheduledMethodRunnable;
import org.springframework.util.ClassUtils;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Schedulerx基于Spring scheduling的配置器
 * @author yaohui
 * @create 2022/8/12 上午10:50
 **/
@Order
public class SchedulerxSchedulingConfigurer implements SchedulingConfigurer, InitializingBean {

    private final Map<String, Pair<Object, Method>> scheduledTaskTargetMap = Maps.newConcurrentMap();

    private final Map<String, Pair<Object, Method>> schedulerXTaskTargetMap = Maps.newConcurrentMap();

    private static SchedulerxSchedulingConfigurer instance = null;

    public static SchedulerxSchedulingConfigurer getInstance() {
        return instance;
    }

    @Autowired
    private NoOpScheduler scheduler;

    @Override
    public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {
        List<Task> scheduledTasks = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(taskRegistrar.getCronTaskList())) {
            scheduledTasks.addAll(taskRegistrar.getCronTaskList());
        }
        if (CollectionUtils.isNotEmpty(taskRegistrar.getFixedDelayTaskList())) {
            scheduledTasks.addAll(taskRegistrar.getFixedDelayTaskList());
        }
        if (CollectionUtils.isNotEmpty(taskRegistrar.getFixedRateTaskList())) {
            scheduledTasks.addAll(taskRegistrar.getFixedRateTaskList());
        }
        if (CollectionUtils.isNotEmpty(taskRegistrar.getTriggerTaskList())) {
            scheduledTasks.addAll(taskRegistrar.getTriggerTaskList());
        }
        for (Task task:scheduledTasks) {
            if (task.getRunnable() instanceof ScheduledMethodRunnable) {
                ScheduledMethodRunnable runnable = (ScheduledMethodRunnable) task.getRunnable();
                register(runnable.getTarget(), runnable.getMethod(), false);
            }
        }
        taskRegistrar.setScheduler(scheduler);
    }

    /**
     * 注册登记任务方法
     * @param target
     * @param method
     */
    public void register(Object target, Method method, boolean isSchedulerX) {
        String className = AopProxyUtils.ultimateTargetClass(target).getName();
        if (ClassUtils.isCglibProxyClassName(className)) {
            className = ClassUtils.getUserClass(target).getName();
        }
        String methodName = method.getName();
        if (isSchedulerX) {
            schedulerXTaskTargetMap.put(className + ":" + methodName, new Pair(target, method));
        } else {
            scheduledTaskTargetMap.put(className + ":" + methodName, new Pair(target, method));
        }
    }

    public Pair<Object, Method> getScheduledTaskTarget(String key) {
        if (schedulerXTaskTargetMap.get(key) != null) {
            return schedulerXTaskTargetMap.get(key);
        } else {
            return scheduledTaskTargetMap.get(key);
        }
    }

    public Collection<Pair<Object, Method>> getSchedulerXTaskTargets() {
        return schedulerXTaskTargetMap.values();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        instance = this;
    }
}
