package com.alibaba.schedulerx.worker.security;

import com.alibaba.schedulerx.common.domain.JSONResult;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.exception.AuthenticateException;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.google.common.collect.Maps;
import com.mashape.unirest.http.HttpResponse;
import com.mashape.unirest.http.JsonNode;
import com.mashape.unirest.http.Unirest;
import com.mashape.unirest.http.exceptions.UnirestException;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.StringUtils;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 *
 * @author 陆昆
 */
public class AliyunAuthenticator implements Authenticator {
    private static final String AUTHENTICATE_DEFAULT = "authenticate_default";
    private static final String AUTHENTICATE_DAUTH = "authenticate_dauth";
    private static final String AUTHENTICATE_RAM = "authenticate_ram";
    private static final String AUTHENTICATE_STS = "authenticate_sts";
    private static final String AUTHENTICATE_URL = "/worker/v1/appgroup/authenticate";
    private static final Logger LOGGER = LogFactory.getLogger(AliyunAuthenticator.class);

    @Override
    public void authenticate(Configuration conf, String namespace, String namespaceSource, List<String> groupIds, List<String> appKeys)
            throws AuthenticateException {
        JSONResult result = JSONResult.geneFailResult();
        String accessKey = StringUtils.EMPTY;
        String secretKey = StringUtils.EMPTY;
        String secretToken = StringUtils.EMPTY;
        
        // -Dschedulerx.appKey优先, 统一在外部获取
//        String appKeyProperty = System.getProperty(WorkerConstants.WORKER_APPKEY);
//        if (StringUtils.isNotBlank(appKeyProperty)) {
//            conf.setProperty(WorkerConstants.APP_KEY, appKeyProperty);
//        }
//        String[] appKeys = conf.getStringArray(WorkerConstants.APP_KEY);
        Map<String, Object> parameterMap = Maps.newHashMap();
        if (StringUtils.isNotBlank(namespace)) {
            parameterMap.put("namespace", namespace);
        }
        if (StringUtils.isNotBlank(namespaceSource)) {
            parameterMap.put("namespaceSource", namespaceSource);
        }
        parameterMap.put("groups", StringUtils.join(groupIds,","));
        
        //appKey鉴权
        if (appKeys != null && appKeys.size() > 0) {
            parameterMap.put("accessKey", StringUtils.join(appKeys,","));
            parameterMap.put("type", AUTHENTICATE_DEFAULT);
        } else {
            //阿里云ak/sk鉴权
            if (StringUtils.isNotBlank(conf.getString(WorkerConstants.ALIYUN_ACESSKEY))) {
                accessKey = conf.getString(WorkerConstants.ALIYUN_ACESSKEY);
                secretKey = conf.getString(WorkerConstants.ALIYUN_SECRETKEY);
                parameterMap.put("type", AUTHENTICATE_RAM);
            } else if (StringUtils.isNotBlank(conf.getString(WorkerConstants.STS_ACESSKEY))) {
                accessKey = conf.getString(WorkerConstants.STS_ACESSKEY);
                secretKey = conf.getString(WorkerConstants.STS_SECRETKEY);
                secretToken = conf.getString(WorkerConstants.STS_TOKEN);
                parameterMap.put("secretToken", secretToken);
                parameterMap.put("type", AUTHENTICATE_STS);
            } else {
                // 从EDAS_ENV_KEY_FILE中获取ak/sk
                String spasKeyFilePath = System.getProperty(WorkerConstants.EDAS_ENV_KEY_FILE);
                if (StringUtils.isEmpty(spasKeyFilePath)) {
                    LOGGER.info("no property of " + WorkerConstants.EDAS_ENV_KEY_FILE);
                } else {
                    InputStream spasIs = null;
                    try {
                        spasIs = new FileInputStream(spasKeyFilePath);
                        Properties properties = new Properties();
                        properties.load(spasIs);
                        
                        if (!WorkerConstants.DOCKER_CREDENTIAL_PATH.equals(spasKeyFilePath)) {
                            if (properties.containsKey(WorkerConstants.ACCESS_KEY)) {
                                accessKey = properties.getProperty(WorkerConstants.ACCESS_KEY);
                            }
                            if (properties.containsKey(WorkerConstants.SECRET_KEY)) {
                                secretKey = properties.getProperty(WorkerConstants.SECRET_KEY);
                            }
                        } else {
                            if (properties.containsKey(WorkerConstants.DOCKER_ACCESS_KEY)) {
                                accessKey = properties.getProperty(WorkerConstants.DOCKER_ACCESS_KEY);
                            }
                            if (properties.containsKey(WorkerConstants.DOCKER_SECRET_KEY)) {
                                secretKey = properties.getProperty(WorkerConstants.DOCKER_SECRET_KEY);
                            }
                        }
                    } catch (Exception e) {
                        LOGGER.error("", e);
                    } finally {
                        if (spasIs != null) {
                            try {
                                spasIs.close();
                            } catch (IOException e) {
                                LOGGER.error("", e);
                            }
                        }
                    }
                    
                    // 默认dauth为edas来源
                    conf.setProperty(WorkerConstants.WORKER_SOURCE, WorkerConstants.WORKER_SOURCE_EDAS);
                    parameterMap.put("type", AUTHENTICATE_DAUTH);
                }
            }
            
            if (StringUtils.isBlank(accessKey)) {
                throw new AuthenticateException("ak is blank");
            }
            if (StringUtils.isBlank(secretKey)) {
                throw new AuthenticateException("sk is blank");
            }
            
            parameterMap.put("accessKey", accessKey.trim());
            parameterMap.put("secretKey", secretKey.trim());
        }
        
        String domain = ConfigUtil.getWorkerConfig().getString(WorkerConstants.WORKER_DOMAIN_NAME);
        if (StringUtils.isNotBlank(domain)) {
            String url = "http://" + domain + AUTHENTICATE_URL;
            try {
                HttpResponse<JsonNode> jsonResponse = Unirest.get(url).queryString(parameterMap).asJson();
                if (jsonResponse != null) {
                    result = JsonUtil.fromJson(jsonResponse.getBody().toString(), JSONResult.class);
                }
            } catch (UnirestException e) {
                LOGGER.error("groupIds: {} authenticate error, url={}, message:{}", groupIds, url, e.getMessage());
                result = JSONResult.geneFailResult(e.getMessage());
            } catch (Exception e) {
                LOGGER.error("groupIds: {} authenticate error, url={}", groupIds, url, e);
                result = JSONResult.geneFailResult(e.getMessage());
            }
        }
        
        if (result != null) {
            if (!result.isSuccess() || !(Boolean)result.getData()) {
                throw new AuthenticateException(result.getMessage()+ " domain=" + domain);
            }
        } else {
            throw new AuthenticateException("authenticate result is null");
        }
    }
    
}
