package com.alibaba.schedulerx.worker.util;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.alibaba.schedulerx.common.domain.ResponseCode;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.exception.TimeOffsetException;

import com.mashape.unirest.http.HttpResponse;
import com.mashape.unirest.http.JsonNode;
import com.mashape.unirest.http.Unirest;
import com.mashape.unirest.http.exceptions.UnirestException;
import org.joda.time.DateTime;

/**
 * Created by yanxun on 2018/5/24.
 *
 * @author yanxun
 * @date 2018/05/24
 */
public class TimeResolver {
    private static TimeResolver INSTANCE = new TimeResolver();
    public static TimeResolver getInstance() {return INSTANCE;}
    private static final String TIME_OFFSET_QUERY_PATH = "/v1/calendar/getOffset";
    private static final String DOMAIN = ConfigUtil.getWorkerConfig().getString(WorkerConstants.WORKER_DOMAIN_NAME);
    private static final String TIME_OFFSET_QUERY_URL = "http://" + DOMAIN + TIME_OFFSET_QUERY_PATH;

    /**
     * query offset accord to baseTimeStamp, if specify calendar, offset compute within calendar.
     * for example, if your calendar is [2018-1-1, 2018-1-3, 2018-3-1], baseTime is 2018-3-1 00:00:30,
     * timeUnit is DAYS, offset is -2, then the result is 2018-1-1 00:00:30, not 2018-2-28 00:00:30.
     * @param calendarName calendar name, null if you don't have calendar
     * @param baseTimeStamp baseTime in millis
     * @param offset offset, negative or positive
     * @param timeUnit offset time unit
     * @return -1 if error occur or offset time (>0) in millis.
     */
    public Date plusTime(String calendarName, long baseTimeStamp, long offset, TimeUnit timeUnit) throws TimeOffsetException,
        UnirestException{
        Map<String, Object> params = new HashMap<>(8);
        if (calendarName != null && !calendarName.trim().isEmpty()) {
            params.put("name", calendarName);
        }
        params.put("baseTimeStamp", baseTimeStamp);
        params.put("offset", offset);
        params.put("timeUnit", timeUnit.toString());
        HttpResponse<JsonNode> response = Unirest.get(TIME_OFFSET_QUERY_URL).queryString(params).asJson();
        if (response.getBody().getObject().getInt("code") != ResponseCode.SUCCESS) {
            throw new TimeOffsetException(response.getBody().getObject().getString("msg"));
        }
        return new Date(response.getBody().getObject().getJSONObject("data").getLong("offsetInMillis"));
    }

    public Date plusDays(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), offset, TimeUnit.DAYS);
    }

    public Date plusHours(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), offset, TimeUnit.HOURS);
    }

    public Date plusMinutes(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), offset, TimeUnit.MINUTES);
    }

    public Date plusSeconds(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), offset, TimeUnit.SECONDS);
    }

    public Date plusMillis(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), offset, TimeUnit.MILLISECONDS);
    }

    public Date minusDays(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), 0-offset, TimeUnit.DAYS);
    }

    public Date minusHours(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), 0-offset, TimeUnit.HOURS);
    }

    public Date minusMinutes(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), 0-offset, TimeUnit.MINUTES);
    }

    public Date minusSeconds(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), 0-offset, TimeUnit.SECONDS);
    }

    public Date minusMillis(String calendarName, Date baseTime, long offset) throws Exception {
        return plusTime(calendarName, baseTime.getTime(), 0-offset, TimeUnit.MILLISECONDS);
    }

    public static void main(String[] args) throws Exception {
        Date baseTime = new DateTime(2018, 3, 6, 0, 0, 30).toDate();
        TimeResolver timeResolver = getInstance();
        System.out.println(timeResolver.plusDays("工作日", baseTime, 2));
        System.out.println(timeResolver.plusDays("工作日", baseTime, -2));
        System.out.println(timeResolver.plusDays(null, baseTime, 4));
        System.out.println(timeResolver.plusDays("工作日", baseTime, 9));

    }
}
