package com.alibaba.schedulerx.worker.actor;

import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.alibaba.schedulerx.common.domain.TaskStatus;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.common.util.IdUtil;
import com.alibaba.schedulerx.protocol.Worker.ContainerReportTaskStatusRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterBatchStartContainersRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterBatchStartContainersResponse;
import com.alibaba.schedulerx.protocol.Worker.MasterDestroyContainerPoolRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterDestroyContainerPoolResponse;
import com.alibaba.schedulerx.protocol.Worker.MasterKillContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterKillContainerResponse;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerResponse;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.batch.ContainerStatusReqHandler;
import com.alibaba.schedulerx.worker.batch.ContainerStatusReqHandlerPool;
import com.alibaba.schedulerx.worker.batch.ReqQueue;
import com.alibaba.schedulerx.worker.container.Container;
import com.alibaba.schedulerx.worker.container.ContainerFactory;
import com.alibaba.schedulerx.worker.container.ContainerPool;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.ClientLoggerMessage;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.pull.PullManager;
import com.alibaba.schedulerx.worker.util.ContanerUtil;
import com.alibaba.schedulerx.worker.util.WorkerIdGenerator;

import akka.actor.ActorSelection;
import akka.actor.Address;
import akka.actor.Props;
import akka.actor.UntypedActor;
import org.apache.commons.lang.StringUtils;

/**
 * @author xiaomeng.hxm
 */
public class ContainerActor extends UntypedActor {
    private ContainerPool containerPool = ContainerFactory.getContainerPool();
    private ContainerStatusReqHandlerPool statusReqBatchHandlerPool = ContainerStatusReqHandlerPool.INSTANCE;
    private static Logger LOGGER = LogFactory.getLogger(ContainerActor.class);
    private static ThreadPoolExecutor containerStarter;
    private boolean enableShareContainerPool = ConfigUtil.getWorkerConfig().getBoolean(WorkerConstants.SHARE_CONTAINER_POOL, false);
    private int batchSize = ConfigUtil.getWorkerConfig().getInt(WorkerConstants.WORKER_MAP_PAGE_SIZE,
        WorkerConstants.WORKER_MAP_PAGE_SIZE_DEFAULT);

    private LogCollector logCollector = LogCollectorFactory.get();

    static {
        containerStarter = new ThreadPoolExecutor(
            8, 8, 30, TimeUnit.SECONDS,
            new LinkedBlockingQueue<Runnable>(), new ThreadFactory() {
            private final AtomicInteger nextId = new AtomicInteger(1);
            private final String namePrefix = "Schedulerx-Container-Starter-Thread-";

            @Override
            public Thread newThread(Runnable r) {
                return new Thread(r, namePrefix + nextId.getAndIncrement());
            }
        }, new ThreadPoolExecutor.CallerRunsPolicy());
        containerStarter.allowCoreThreadTimeOut(true);
    }

    public static Props props() {
        return Props.create(ContainerActor.class);
    }

    @Override
    public void onReceive(Object obj) throws Throwable {
        if (obj instanceof MasterStartContainerRequest) {
            handleStartContainer((MasterStartContainerRequest)obj);
        } else if (obj instanceof MasterBatchStartContainersRequest) {
            handleBatchStartContainers((MasterBatchStartContainersRequest)obj);
        } else if (obj instanceof MasterKillContainerRequest) {
            handleKillContainer((MasterKillContainerRequest)obj);
        } else if (obj instanceof MasterDestroyContainerPoolRequest) {
            handleDestroyContainerPool((MasterDestroyContainerPoolRequest)obj);
        }
    }

    private void handleStartContainer(MasterStartContainerRequest request) {
        MasterStartContainerResponse response = null;
        try {
            String uniqueId = startContainer(request);
            response = MasterStartContainerResponse.newBuilder()
                .setSuccess(true)
                .build();
            LOGGER.debug("submit container to containerPool, uniqueId={}, cost={}ms", uniqueId,
                (System.currentTimeMillis() - request.getScheduleTime()));
        } catch (Throwable e) {
            String uniqueId = IdUtil.getUniqueId(request.getJobId(), request.getJobInstanceId(), request.getTaskId());
            response = MasterStartContainerResponse.newBuilder()
                .setSuccess(false)
                .setMessage(ExceptionUtil.getMessage(e))
                .build();
            LOGGER.error("handleStartContainer error.", e);
            logCollector.collect(uniqueId, ClientLoggerMessage.CONTAINER_START_FAIL, e);
        }
        getSender().tell(response, getSelf());
    }

    private void handleBatchStartContainers(MasterBatchStartContainersRequest request) {
        MasterBatchStartContainersResponse response;
        LOGGER.info("jobInstanceId={}, batch start containers, size:{}", request.getJobInstanceId(),
            request.getStartReqsCount());
        containerStarter.submit(new ContainerStartRunnable(request));
        response = MasterBatchStartContainersResponse.newBuilder()
            .setSuccess(true)
            .build();
        getSender().tell(response, getSelf());
    }

    private String startContainer(MasterStartContainerRequest request) throws Exception {
        String uniqueId = IdUtil.getUniqueId(request.getJobId(), request.getJobInstanceId(), request.getTaskId());
        LOGGER.debug("handleStartContainer, uniqueId={}, cost={}ms", uniqueId,
            (System.currentTimeMillis() - request.getScheduleTime()));

        JobContext context = ContanerUtil.convert2JobContext(request);
        Container container = ContainerFactory.create(context);
        if (container != null) {
            Object jobInstanceLock = containerPool.getInstanceLock(request.getJobInstanceId());
            synchronized (jobInstanceLock) {
                containerPool.put(context.getUniqueId(), container);
                //是否共享containerPool，如果共享，statusReqBatchHandlerPool只有一个key=0的hander
                long statusReqBatchHandlerKey = (enableShareContainerPool ? 0 :request.getJobInstanceId());
                if (!statusReqBatchHandlerPool.contains(statusReqBatchHandlerKey)) {
                    ReqQueue<ContainerReportTaskStatusRequest> reqQueue = new ReqQueue<>(statusReqBatchHandlerKey,
                        10 * 10000);
                    reqQueue.init();
                    statusReqBatchHandlerPool.start(statusReqBatchHandlerKey, new ContainerStatusReqHandler<>(
                            statusReqBatchHandlerKey, 1, 1, batchSize,
                        reqQueue, request.getInstanceMasterAkkaPath()));
                }

                int consumerNum = request.hasConsumerNum() ? request.getConsumerNum()
                    : WorkerConstants.CONSUMER_NUM_DEFAULT;
                containerPool.submit(context.getJobId(), context.getJobInstanceId(), context.getTaskId(), container,
                    consumerNum);
            }
        } else {
            LOGGER.warn("Container is null, uniqueId={}", uniqueId);
        }
        return uniqueId;
    }

    private void handleKillContainer(MasterKillContainerRequest request) {
        MasterKillContainerResponse response;
        long jobId = request.getJobId();
        long jobInstanceId = request.getJobInstanceId();
        String uniqueIdToLogger = StringUtils.EMPTY;
        try {
            if (request.hasTaskId()) {
                //kill task container
                long taskId = request.getTaskId();
                String uniqueId = IdUtil.getUniqueId(jobId, jobInstanceId, taskId);
                if (containerPool.contain(uniqueId)) {
                    containerPool.get(uniqueId).kill();
                }
                uniqueIdToLogger = uniqueId;
                LOGGER.info("kill task container success, uniqueId={}", uniqueIdToLogger);
            } else {
                uniqueIdToLogger = IdUtil.getUniqueIdWithoutTask(jobId, jobInstanceId);
                //kill instance container pool
                killInstance(jobId, jobInstanceId);
                LOGGER.info("kill instance success, uniqueId:{}", uniqueIdToLogger);
            }
            response = MasterKillContainerResponse.newBuilder()
                .setSuccess(true)
                .build();
            logCollector.collect(uniqueIdToLogger, ClientLoggerMessage.CONTAINER_KILL_SUCCESS);
        } catch (Throwable t) {
            LOGGER.error("kill container exception", t);
            logCollector.collect(uniqueIdToLogger, ClientLoggerMessage.CONTAINER_KILL_FAIL, t);
            response = MasterKillContainerResponse.newBuilder()
                .setSuccess(false)
                .setMessage(t.getMessage())
                .build();
        }
        getSender().tell(response, getSelf());
    }

    private void handleDestroyContainerPool(MasterDestroyContainerPoolRequest request) {
        if (!enableShareContainerPool) {
            try {

                ContainerStatusReqHandler<ContainerReportTaskStatusRequest> handler = statusReqBatchHandlerPool
                    .getHandlers()
                    .get(request.getJobInstanceId());
    
                if (handler != null) {
                    Object jobInstanceLock = containerPool.getInstanceLock(request.getJobInstanceId());
                    synchronized (jobInstanceLock) {
                        if (handler.getLatestRequest() != null && handler.getLatestRequest().getSerialNum() != request
                                .getSerialNum()) {
                            LOGGER.info("skip handleDestroyContainerPool cycleId={}_{}, handler serialNum={}.",
                                    request.getJobInstanceId(),
                                    request.getSerialNum(), handler.getLatestRequest().getSerialNum());
                            return;
                        }
                        LOGGER.info("handleDestroyContainerPool from cycleId={}_{}, handler serialNum={}.",
                                request.getJobInstanceId(), request.getSerialNum(), handler.getLatestRequest().getSerialNum());

                        containerPool.destroyByInstance(request.getJobInstanceId());
                        statusReqBatchHandlerPool.stop(request.getJobInstanceId());
                        PullManager.INSTANCE.stop(request.getJobInstanceId());
                    }
                }
            } catch (Throwable th) {
                LOGGER.error("cycleId={}_{} handleDestroyContainerPool failed.", request.getJobInstanceId(),
                    request.getSerialNum(), th);
            } finally {
                MasterDestroyContainerPoolResponse response = MasterDestroyContainerPoolResponse.newBuilder()
                    .setSuccess(true)
                    .setDeliveryId(request.getDeliveryId())
                    .build();
                getSender().tell(response, getSelf());
            }
        }
        containerPool.releaseInstanceLock(request.getJobInstanceId());
    }

    private void killInstance(long jobId, long jobInstanceId) {
        Map<String, Container> containerMap = containerPool.getContainerMap();
        String prefixKey = jobId + IdUtil.SPLITTER_TOKEN + jobInstanceId;
        Iterator<Entry<String, Container>> it = containerMap.entrySet().iterator();
        while (it.hasNext()) {
            Entry<String, Container> entry = it.next();
            String uniqueId = entry.getKey();
            Container container = entry.getValue();
            if (uniqueId.startsWith(prefixKey)) {
                container.kill();
                it.remove();
            }
        }
        if (!enableShareContainerPool) {
            containerPool.destroyByInstance(jobInstanceId);
        }
    }

    private class ContainerStartRunnable implements Runnable {
        private MasterBatchStartContainersRequest request;

        ContainerStartRunnable(MasterBatchStartContainersRequest request) {
            this.request = request;
        }

        @Override
        public void run() {
            for (MasterStartContainerRequest req : request.getStartReqsList()) {
                try {
                    String uniqueId = startContainer(req);
                    LOGGER.debug("submit container to containerPool, uniqueId={}, cost={}ms", uniqueId,
                        (System.currentTimeMillis() - req.getScheduleTime()));
                } catch (Throwable e) {
                    String uniqueId = IdUtil.getUniqueId(req.getJobId(), request.getJobInstanceId(), req.getTaskId());
                    logCollector.collect(uniqueId, ClientLoggerMessage.CONTAINER_START_FAIL, e);
                    // report task fail status to task master
                    ContainerReportTaskStatusRequest.Builder resultBuilder = ContainerReportTaskStatusRequest
                        .newBuilder();
                    resultBuilder.setJobId(req.getJobId());
                    resultBuilder.setJobInstanceId(req.getJobInstanceId());
                    resultBuilder.setTaskId(req.getTaskId());
                    resultBuilder.setStatus(TaskStatus.FAILED.getValue());
                    Address address = SchedulerxWorker.actorSystem.provider().getDefaultAddress();
                    String workerAddr = address.host().get() + ":" + address.port().get();
                    resultBuilder.setWorkerAddr(workerAddr);
                    resultBuilder.setWorkerId(WorkerIdGenerator.get());
                    if (req.getTaskName() != null) {
                        resultBuilder.setTaskName(req.getTaskName());
                    }
                    ActorSelection masterActorSelection = SchedulerxWorker.actorSystem.actorSelection(req.getInstanceMasterAkkaPath());
                    masterActorSelection.tell(resultBuilder.build(), null);
                }
            }
        }
    }

}
