package com.alibaba.schedulerx.worker.actor;

import java.util.Collections;
import java.util.List;

import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.domain.JobInstanceData;
import com.alibaba.schedulerx.common.domain.JobInstanceInfo;
import com.alibaba.schedulerx.common.domain.JobInstanceInfo.JobInstanceInfoBuilder;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.common.util.IdUtil;
import com.alibaba.schedulerx.common.util.ManagementUtil;
import com.alibaba.schedulerx.protocol.Common.UpstreamData;
import com.alibaba.schedulerx.protocol.Server.ServerCheckTaskMasterRequest;
import com.alibaba.schedulerx.protocol.Server.ServerCheckTaskMasterResponse;
import com.alibaba.schedulerx.protocol.Server.ServerKillJobInstanceRequest;
import com.alibaba.schedulerx.protocol.Server.ServerKillJobInstanceResponse;
import com.alibaba.schedulerx.protocol.Server.ServerKillTaskRequest;
import com.alibaba.schedulerx.protocol.Server.ServerKillTaskResponse;
import com.alibaba.schedulerx.protocol.Server.ServerPushLogConfigRequest;
import com.alibaba.schedulerx.protocol.Server.ServerRetryTasksRequest;
import com.alibaba.schedulerx.protocol.Server.ServerRetryTasksResponse;
import com.alibaba.schedulerx.protocol.Server.ServerSubmitJobInstanceRequest;
import com.alibaba.schedulerx.protocol.Server.ServerSubmitJobInstanceResponse;
import com.alibaba.schedulerx.protocol.Server.ServerThreadDumpRequest;
import com.alibaba.schedulerx.protocol.Server.ServerThreadDumpResponse;
import com.alibaba.schedulerx.protocol.Worker.MasterNotifyWorkerPullRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterNotifyWorkerPullResponse;
import com.alibaba.schedulerx.protocol.Worker.WorkerReportJobInstanceStatusRequest;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.log.appender.LogConfig;
import com.alibaba.schedulerx.worker.logcollector.ClientLoggerMessage;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.master.TaskMaster;
import com.alibaba.schedulerx.worker.master.TaskMasterFactory;
import com.alibaba.schedulerx.worker.master.TaskMasterPool;
import com.alibaba.schedulerx.worker.pull.PullManager;

import akka.actor.UntypedActor;
import com.google.common.collect.Lists;
import org.joda.time.DateTime;

/**
 *
 * @author xiaomeng.hxm
 */
public class JobInstanceActor extends UntypedActor {
    private TaskMasterPool masterPool = TaskMasterPool.INSTANCE;
    private LogCollector logCollector = LogCollectorFactory.get();
    private static final Logger LOGGER = LogFactory.getLogger(JobInstanceActor.class);

    @Override
    public void onReceive(Object obj) throws Throwable {
        if (obj instanceof ServerSubmitJobInstanceRequest) {
            handleSubmitJobInstance((ServerSubmitJobInstanceRequest)obj);
        } else if (obj instanceof ServerKillJobInstanceRequest) {
            handleKillJobInstance((ServerKillJobInstanceRequest)obj);
        }  else if (obj instanceof ServerRetryTasksRequest) {
            handleRetryTasks((ServerRetryTasksRequest) obj);
        } else if (obj instanceof ServerKillTaskRequest) {
            handleKillTask((ServerKillTaskRequest) obj);
        } else if (obj instanceof ServerCheckTaskMasterRequest) {
            handCheckTaskMaster((ServerCheckTaskMasterRequest) obj);
        } else if (obj instanceof MasterNotifyWorkerPullRequest) {
            handleInitPull((MasterNotifyWorkerPullRequest) obj);
        } else if (obj instanceof ServerThreadDumpRequest) {
            handleThreadDump((ServerThreadDumpRequest) obj);
        } else if (obj instanceof ServerPushLogConfigRequest) {
            handlePushLogConfig((ServerPushLogConfigRequest) obj);
        }
    }

    private void handleSubmitJobInstance(ServerSubmitJobInstanceRequest request) {
        LOGGER.info("handleSubmitJobInstance, jobInstanceId=" + request.getJobInstanceId());
        ServerSubmitJobInstanceResponse response = null;

        if (masterPool.contains(request.getJobInstanceId())) {
            String errMsg = "jobInstanceId=" + request.getJobInstanceId() + " is still running!";
            LOGGER.debug(errMsg);
            logCollector.collect(IdUtil.getUniqueIdWithoutTask(request.getJobId(),request.getJobInstanceId()),
                ClientLoggerMessage.appendMessage(ClientLoggerMessage.INSTANCE_TRIGGER_FAIL, errMsg));
            response = ServerSubmitJobInstanceResponse.newBuilder()
                    .setSuccess(false)
                    .setMessage(errMsg)
                    .build();
            getSender().tell(response, getSelf());
        } else {
            response = ServerSubmitJobInstanceResponse.newBuilder()
                    .setSuccess(true)
                    .build();
            getSender().tell(response, getSelf());

            try {
                JobInstanceInfo jobInstanceInfo = convet2JobInstanceInfo(request);
                TaskMaster taskMaster = TaskMasterFactory.create(jobInstanceInfo, getContext());
                masterPool.put(jobInstanceInfo.getJobInstanceId(), taskMaster);
                taskMaster.submitInstance(jobInstanceInfo);
                LOGGER.debug("submit jobInstanceId={} successfully", request.getJobInstanceId());
                logCollector.collect(IdUtil.getUniqueIdWithoutTask(request.getJobId(),request.getJobInstanceId()),
                        ClientLoggerMessage.INSTANCE_TRIGGER_SUCCESS);
            } catch (Throwable e) {
                LOGGER.error("handleSubmitJobInstance error, jobInstanceId={}, ", request.getJobInstanceId(), e);
                logCollector.collect(IdUtil.getUniqueIdWithoutTask(request.getJobId(),request.getJobInstanceId()),
                        ClientLoggerMessage.INSTANCE_TRIGGER_FAIL, e);
                WorkerReportJobInstanceStatusRequest req = WorkerReportJobInstanceStatusRequest.newBuilder()
                    .setJobId(request.getJobId())
                    .setJobInstanceId(request.getJobInstanceId())
                    .setStatus(InstanceStatus.FAILED.getValue())
                    .setResult(ExceptionUtil.getMessage(e))
                    .setGroupId(request.getGroupId())
                    .build();
                SchedulerxWorker.AtLeastDeliveryRoutingActor.tell(req, null);
            }
        }
    }

    private void handleKillJobInstance(ServerKillJobInstanceRequest request) {
        LOGGER.info("handleKillJobInstance, jobInstanceId=" + request.getJobInstanceId());
        ServerKillJobInstanceResponse response = null;
        String uniqueId = IdUtil.getUniqueIdWithoutTask(request.getJobId(),request.getJobInstanceId());
        try {
            long jobInstanceId = request.getJobInstanceId();
            if (!masterPool.contains(jobInstanceId)) {
                response = ServerKillJobInstanceResponse.newBuilder()
                    .setSuccess(false)
                    .setMessage(jobInstanceId + " is not exist")
                    .build();
                logCollector.collect(uniqueId, ClientLoggerMessage.appendMessage(ClientLoggerMessage.INSTANCE_KILL_START_FAIL, response.getMessage()));
            } else {
                masterPool.get(jobInstanceId).killInstance("killed from server");
                response = ServerKillJobInstanceResponse.newBuilder().setSuccess(true).build();
                logCollector.collect(uniqueId, ClientLoggerMessage.INSTANCE_KILL_START_SUCCESS);
            }
        } catch (Throwable e) {
            LOGGER.error("[JobInstanceActor]handleKillJobInstance error, uniqueId:{}",uniqueId, e);
            response = ServerKillJobInstanceResponse.newBuilder()
                .setSuccess(false)
                .setMessage(ExceptionUtil.getMessage(e))
                .build();
        } finally {
            getSender().tell(response, getSelf());
        }
    }

    private void handleKillTask(ServerKillTaskRequest request) {
        LOGGER.info("handleKillTask, jobInstanceId=" + request.getJobInstanceId());
        ServerKillTaskResponse response = null;
        try {
            long jobInstanceId = request.getJobInstanceId();
            if (!masterPool.contains(jobInstanceId)) {
                response = ServerKillTaskResponse.newBuilder()
                    .setSuccess(false)
                    .setMessage(jobInstanceId + " is not exist")
                    .build();
            } else {
                masterPool.get(jobInstanceId).killTask(IdUtil.getUniqueId(request.getJobId(), request.getJobInstanceId(), request.getTaskId()),
                    request.getWorkerId(), request.getWorkerAddr());
                response = ServerKillTaskResponse.newBuilder().setSuccess(true).build();
            }
        } catch (Throwable e) {
            LOGGER.error("", e);
            response = ServerKillTaskResponse.newBuilder()
                .setSuccess(false)
                .setMessage(ExceptionUtil.getMessage(e))
                .build();
        } finally {
            getSender().tell(response, getSelf());
        }
    }

    //private void handleTaskStatus(ContainerReportTaskStatusRequest request) {
    //    //ContainerReportTaskStatusResponse response = null;
    //    TaskMaster taskMaster = masterPool.get(request.getJobInstanceId());
    //    LOGGER.debug("handleTaskStatus, uniqueId:{}, status:{}, workerAddr:{}",
    //        IdUtil.getUniqueId(request.getJobId(), request.getJobInstanceId(), request.getTaskId()),
    //        request.getStatus(), request.getWorkerAddr());
    //    if (taskMaster != null) {
    //        taskMaster.updateTaskStatus(request);
    //    }
    //    //response = ContainerReportTaskStatusResponse.newBuilder()
    //    //    .setSuccess(true)
    //    //    .build();
    //    //getSender().tell(response, getSelf());
    //}
    //
    //private void handleBatchTaskStatues(ContainerBatchReportTaskStatuesRequest request) {
    //    for (ContainerReportTaskStatusRequest req : request.getTaskStatuesList()) {
    //        handleTaskStatus(req);
    //    }
    //}
    //
    //private void handleMapTask(ExecutorMapTaskRequest request) throws Exception {
    //    WorkerMapTaskResponse response = null;
    //    try {
    //        long jobInstanceId = request.getJobInstanceId();
    //        TaskMaster taskMaster = masterPool.get(jobInstanceId);
    //        if (taskMaster != null) {
    //            if (!(taskMaster instanceof MapTaskMaster)) {
    //                response = WorkerMapTaskResponse.newBuilder().setSuccess(false)
    //                        .setMessage("TaskMaster is not MapTaskMaster")
    //                        .build();
    //                taskMaster.updateNewInstanceStatus(InstanceStatus.FAILED, "TaskMaster is not MapTaskMaster");
    //            } else {
    //                try {
    //                    ((MapTaskMaster)taskMaster).map(request.getTaskBodyList(), request.getTaskName());
    //                    response = WorkerMapTaskResponse.newBuilder().setSuccess(true).build();
    //                } catch (Exception e) {
    //                    taskMaster.updateNewInstanceStatus(InstanceStatus.FAILED, ExceptionUtil.getMessage(e));
    //                    throw e;
    //                }
    //            }
    //        } else {
    //            response = WorkerMapTaskResponse.newBuilder().setSuccess(false)
    //                    .setMessage("can't found TaskMaster by jobInstanceId=" + jobInstanceId)
    //                    .build();
    //        }
    //    } catch (Exception e) {
    //        response = WorkerMapTaskResponse.newBuilder().setSuccess(false)
    //                .setMessage(ExceptionUtil.getMessage(e))
    //                .build();
    //    } finally {
    //        getSender().tell(response, getSelf());
    //    }
    //}

    private void handleRetryTasks(ServerRetryTasksRequest request) {
        JobInstanceInfo jobInstanceInfo = convet2JobInstanceInfo(request);
        ServerRetryTasksResponse response = null;
        TaskMaster taskMaster = masterPool.get(jobInstanceInfo.getJobInstanceId());
        if (taskMaster == null) {
            try {
                taskMaster = TaskMasterFactory.create(jobInstanceInfo, getContext());
                masterPool.put(jobInstanceInfo.getJobInstanceId(), taskMaster);
            } catch (Exception e) {
                LOGGER.error("", e);
                response = ServerRetryTasksResponse.newBuilder()
                        .setSuccess(false)
                        .setMessage(ExceptionUtil.getMessage(e))
                        .build();
            }
        }
        if (taskMaster != null) {
            taskMaster.retryTasks(request.getRetryTaskEntityList());
            response = ServerRetryTasksResponse.newBuilder().setSuccess(true).build();
        }

        getSender().tell(response, getSelf());
    }

    private void handCheckTaskMaster(ServerCheckTaskMasterRequest request) {
        long jobInstanceId = request.getJobInstanceId();
        ServerCheckTaskMasterResponse response = null;
        if (!masterPool.contains(jobInstanceId)) {
            response = ServerCheckTaskMasterResponse.newBuilder()
                    .setSuccess(false)
                    .setMessage("TaskMaster is not existed of jobInstance=" + jobInstanceId)
                    .build();
        } else {
            response = ServerCheckTaskMasterResponse.newBuilder().setSuccess(true).build();
        }
        getSender().tell(response, getSelf());
    }

    private void handleInitPull(MasterNotifyWorkerPullRequest request) {
        MasterNotifyWorkerPullResponse response = null;
        try {
            PullManager.INSTANCE.init(request.getJobInstanceId(), request.getPageSize(),
                    request.getQueueSize(), request.getConsumerSize(), request.getTaskMasterAkkaPath());
            response = MasterNotifyWorkerPullResponse.newBuilder().setSuccess(true).build();
        } catch (Exception e) {
            LOGGER.error("", e);
            response = MasterNotifyWorkerPullResponse.newBuilder().setSuccess(false)
                    .setMessage(ExceptionUtil.getMessage(e))
                    .build();
        } finally {
            getSender().tell(response, getSelf());
        }
    }
    
    private void handleThreadDump(ServerThreadDumpRequest request) {
        ServerThreadDumpResponse response = null;
        try {
            long jobId = request.getJobId();
            long jobInstanceId = request.getJobInstanceId();
            boolean enableShareContainerPool = ConfigUtil.getWorkerConfig().getBoolean(WorkerConstants.SHARE_CONTAINER_POOL, false);
            String threadName = null;
            if (enableShareContainerPool) {
                threadName = "Schedulerx-Shared-Container-Thread";
            } else {
                threadName = "Schedulerx-Container-Thread-" + jobId + "_" + jobInstanceId;
            }
            List<String> resultLines = ManagementUtil.getStackTraces(threadName);
            response = ServerThreadDumpResponse.newBuilder().setSuccess(true)
                .addAllLine(resultLines).build();
        } catch (Exception e) {
            LOGGER.error("", e);
            response = ServerThreadDumpResponse.newBuilder().setSuccess(false)
                    .setMessage(ExceptionUtil.getMessage(e))
                    .build();
        } finally {
            getSender().tell(response, getSelf());
        }
    }
    
    private void handlePushLogConfig(ServerPushLogConfigRequest request) {
        LogConfig.INSTANCE.setEnable(request.getEnable());
    }

    private JobInstanceInfo convet2JobInstanceInfo(ServerSubmitJobInstanceRequest request) {
        List<String> workers =  Lists.newCopyOnWriteArrayList(request.getWorkersList());
        Collections.shuffle(workers);
        JobInstanceInfoBuilder builder = JobInstanceInfo.newBuilder();
        builder.setJobId(request.getJobId());
        builder.setJobInstanceId(request.getJobInstanceId());
        builder.setExecuteMode(request.getExecuteMode());
        builder.setJobType(request.getJobType());
        builder.setContent(request.getContent());
        builder.setUser(request.getUser());
        builder.setScheduleTime(new DateTime(request.getScheduleTime()));
        builder.setDataTime(new DateTime(request.getDataTime()));
        builder.setAllWorkers(workers);
        builder.setJobConcurrency(request.getJobConcurrency());
        builder.setRegionId(request.getRegionId());
        builder.setAppGroupId(request.getAppGroupId());
        builder.setTimeType(request.hasTimeType() ? request.getTimeType() : 0);
        builder.setTimeExpression(request.hasTimeExpression() ? request.getTimeExpression() : null);
        builder.setGroupId(request.getGroupId());
        if (request.hasParameters()) {
            builder.setParameters(request.getParameters());
        }
        if (request.hasXattrs()) {
            builder.setXattrs(request.getXattrs());
        }
        if (request.hasInstanceParameters()) {
            builder.setInstanceParameters(request.getInstanceParameters());
        }
        List<UpstreamData> upstreamDataList = request.getUpstreamDataList();
        List<JobInstanceData> upstreamDataPairList = Lists.newArrayList();
        for (UpstreamData upstreamData : upstreamDataList) {
            upstreamDataPairList.add(new JobInstanceData(upstreamData.getJobName(), upstreamData.getData()));
        }
        builder.setUpstreamData(upstreamDataPairList);
        if (request.hasMaxAttempt()) {
            builder.setMaxAttempt(request.getMaxAttempt());
        }
        if (request.hasAttempt()) {
            builder.setAttempt(request.getAttempt());
        }
        if (request.hasWfInstanceId()) {
            builder.setWfInstanceId(request.getWfInstanceId());
        }
        if (request.hasJobName()) {
            builder.setJobName(request.getJobName());
        }

        return builder.build();
    }

    private JobInstanceInfo convet2JobInstanceInfo(ServerRetryTasksRequest request) {
        List<String> workers =  Lists.newCopyOnWriteArrayList(request.getWorkersList());
        Collections.shuffle(workers);
        JobInstanceInfoBuilder builder = JobInstanceInfo.newBuilder();
        builder.setJobId(request.getJobId());
        builder.setJobInstanceId(request.getJobInstanceId());
        builder.setExecuteMode(request.getExecuteMode());
        builder.setJobType(request.getJobType());
        builder.setContent(request.getContent());
        builder.setUser(request.getUser());
        builder.setScheduleTime(new DateTime(request.getScheduleTime()));
        builder.setDataTime(new DateTime(request.getDataTime()));
        builder.setAllWorkers(workers);
        builder.setJobConcurrency(request.getJobConcurrency());
        builder.setRegionId(request.getRegionId());
        builder.setAppGroupId(request.getAppGroupId());
        builder.setTimeType(request.hasTimeType() ? request.getTimeType() : 0);
        builder.setTimeExpression(request.hasTimeExpression() ? request.getTimeExpression() : null);
        builder.setGroupId(request.getGroupId());
        if (request.hasParameters()) {
            builder.setParameters(request.getParameters());
        }
        if (request.hasXattrs()) {
            builder.setXattrs(request.getXattrs());
        }
        if (request.hasInstanceParameters()) {
            builder.setInstanceParameters(request.getInstanceParameters());
        }
        List<UpstreamData> upstreamDataList = request.getUpstreamDataList();
        List<JobInstanceData> upstreamDataPairList = Lists.newArrayList();
        for (UpstreamData upstreamData : upstreamDataList) {
            upstreamDataPairList.add(new JobInstanceData(upstreamData.getJobName(), upstreamData.getData()));
        }
        builder.setUpstreamData(upstreamDataPairList);
        if (request.hasMaxAttempt()) {
            builder.setMaxAttempt(request.getMaxAttempt());
        }
        if (request.hasAttempt()) {
            builder.setAttempt(request.getAttempt());
        }
        if (request.hasWfInstanceId()) {
            builder.setWfInstanceId(request.getWfInstanceId());
        }
        if (request.hasJobName()) {
            builder.setJobName(request.getJobName());
        }

        return builder.build();
    }

}
