package com.alibaba.schedulerx.worker.batch;

import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;

import com.google.common.collect.Lists;

/**
 * Queue-based load leveling buffer, used for parallel/grid reqs buffer.
 * Can buffer any reqs.
 * @author yanxun on 2019/1/5.
 */
public class ReqQueue<T> {
    private long jobInstanceId;
    private int capacity;
    private static final Logger LOGGER = LogFactory.getLogger(ReqQueue.class);

    private BlockingQueue<T> requests;

    public ReqQueue(long jobInstanceId, int capacity) {
        this.jobInstanceId = jobInstanceId;
        this.capacity = capacity;
    }

    public void init() {
        requests = new LinkedBlockingQueue<>(capacity);
    }

    public void submitRequest(T request) throws Exception {
        try {
            if (request != null) {
                requests.put(request);
            }
        } catch (Throwable e) {
            LOGGER.error("add task status request to queue error, jobInstanceId:{}",
                jobInstanceId, e);
            throw e;
        }
    }

    /**
     * return requests with no more than #batchSize elements.
     * @param batchSize max number of elements attempt to retrieve
     * @return requests list
     */
    public List<T> retrieveRequests(int batchSize) {
        List<T> res = Lists.newLinkedList();
        T request;
        for (int i=0; i<batchSize; i++) {
            request = requests.poll();
            if (request == null) {
                //empty, just break
                break;
            }
            res.add(request);
        }
        return res;
    }

    public void clear() {
        requests.clear();
    }

    public int size() {
        return requests.size();
    }

    public void setCapacity(int capacity) {
        this.capacity = capacity;
    }
}
