package com.alibaba.schedulerx.worker.container;

import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;

import com.google.common.collect.Maps;

/**
 *
 * @author xiaomeng.hxm
 */
public class ThreadContainerPool extends ContainerPool {

    private static ThreadContainerPool instance = new ThreadContainerPool();
    private ThreadContainerPool() {}
    public static ThreadContainerPool getInstance() {
        return instance;
    }

    private Map<Long, ExecutorService> threadPoolMap = Maps.newConcurrentMap();
    private ThreadLocal<JobContext> contextThreadLocal = new ThreadLocal<JobContext>();
    private static boolean enableShareContainerPool = ConfigUtil.getWorkerConfig().getBoolean(WorkerConstants.SHARE_CONTAINER_POOL, false);
    private static ThreadPoolExecutor SHARED_THREAD_POOL;

    private static final Map<Long, Object> jobInstanceLockMap = Maps.newConcurrentMap();

    static {
        if (enableShareContainerPool) {
            int poolSize = ConfigUtil.getWorkerConfig().getInt(WorkerConstants.SHARE_POOL_SIZE, WorkerConstants.SHARE_POOL_SIZE_DEFAULT); 
            SHARED_THREAD_POOL = new ThreadPoolExecutor(poolSize, poolSize,
                    30, TimeUnit.SECONDS,
                    new LinkedBlockingQueue<Runnable>(10240),
                    new ThreadFactory() {
                        int index = 0;
                        @Override
                        public Thread newThread(Runnable runnable) {
                            return new Thread(runnable, "Schedulerx-Shared-Container-Thread-" + (index++));
                        }
                    });
        }
    }

    @Override
    public void submit(final long jobId, final long jobInstanceId, long taskId, Container container, int consumerSize) {
        if (!enableShareContainerPool) {
            if (!threadPoolMap.containsKey(jobInstanceId)) {
                synchronized (this) {
                    if (!threadPoolMap.containsKey(jobInstanceId)) {
                        ThreadPoolExecutor threadPool = new ThreadPoolExecutor(consumerSize, consumerSize,
                                30, TimeUnit.SECONDS,
                                new LinkedBlockingQueue<Runnable>(),
                                new ThreadFactory() {
                                    int index = 0;
                                    @Override
                                    public Thread newThread(Runnable runnable) {
                                        return new Thread(runnable, "Schedulerx-Container-Thread-" 
                                            + jobId + "_" + jobInstanceId + "-" + (index++) );
                                    }
                                });
                        threadPoolMap.put(jobInstanceId, threadPool);
                        threadPool.allowCoreThreadTimeOut(true);
                    }
                }
            }
            threadPoolMap.get(jobInstanceId).execute((Runnable) container);
        } else {
            SHARED_THREAD_POOL.execute((Runnable) container);
        }
    }

    @Override
    public synchronized boolean destroyByInstance(long jobInstanceId) {
        if (threadPoolMap.containsKey(jobInstanceId)) {
            threadPoolMap.get(jobInstanceId).shutdownNow();
            threadPoolMap.remove(jobInstanceId);
        }
        return true;
    }

    @Override
    public void setContext(JobContext jobContext) {
        contextThreadLocal.set(jobContext);
    }

    @Override
    public JobContext getContext() {
        return contextThreadLocal.get();
    }

    @Override
    public void removeContext() {
        contextThreadLocal.remove();
    }

    @Override
    public Object getInstanceLock(long jobInstanceId) {
        Object lock = jobInstanceLockMap.get(Long.valueOf(jobInstanceId));
        if(lock == null){
            synchronized (this){
                lock = jobInstanceLockMap.get(Long.valueOf(jobInstanceId));
                if(lock == null){
                    lock = new Object();
                    jobInstanceLockMap.put(Long.valueOf(jobInstanceId), lock);
                }
            }
        }
        return lock;
    }

    @Override
    public void releaseInstanceLock(long jobInstanceId) {
        jobInstanceLockMap.remove(Long.valueOf(jobInstanceId));
    }

}
