package com.alibaba.schedulerx.worker.master;

import java.io.IOException;
import java.util.List;

import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.domain.JobInstanceInfo;
import com.alibaba.schedulerx.common.domain.TaskStatus;
import com.alibaba.schedulerx.common.domain.TimeType;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.IdUtil;
import com.alibaba.schedulerx.common.util.JobUtil;
import com.alibaba.schedulerx.protocol.Worker.MasterDestroyContainerPoolRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterKillContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterKillContainerResponse;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerResponse;
import com.alibaba.schedulerx.protocol.utils.FutureUtils;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.ClientLoggerMessage;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.processor.JobProcessor;
import com.alibaba.schedulerx.worker.processor.MapJobProcessor;
import com.alibaba.schedulerx.worker.util.ActorPathUtil;
import com.alibaba.schedulerx.worker.util.JavaProcessorProfileUtil;

import akka.actor.ActorContext;
import akka.actor.ActorSelection;


/**
 *
 * @author xiaomeng.hxm
 */
public class StandaloneTaskMaster extends TaskMaster {

    private LogCollector logCollector = LogCollectorFactory.get();
    private static final Logger LOGGER = LogFactory.getLogger(StandaloneTaskMaster.class);

    private ActorSelection currentSelection = null;

    public StandaloneTaskMaster(JobInstanceInfo jobInstanceInfo, ActorContext actorContext) throws Exception {
        super(jobInstanceInfo, actorContext);
        currentSelection = getActorContext().actorSelection(getLocalContainerRouterPath());
    }

    /**
     * Standalone TaskMaster only submit to localhost
     */
    @Override
    public synchronized void submitInstance(JobInstanceInfo info) {
        long taskId = aquireTaskId();
        String uniqueId = IdUtil.getUniqueId(info.getJobId(), info.getJobInstanceId(), taskId);
        MasterStartContainerRequest.Builder builder = convert2StartContainerRequestBuilder(info, taskId);
        MasterStartContainerRequest request = builder.build();
        try {
            // 如果秒级别任务开启了任务执行分发，则轮询选取执行机器
            Boolean dispatch = ConfigUtil.getWorkerConfig().getBoolean(WorkerConstants.SECOND_DELAY_STANDALONE_DISPATCH,
                    WorkerConstants.SECOND_DELAY_STANDALONE_DISPATCH_DEFAULT);
            if(JobUtil.isSecondTypeJob(TimeType.parseValue(jobInstanceInfo.getTimeType())) && dispatch){
                currentSelection = getActorContext().actorSelection(ActorPathUtil.getContainerRouterPath(selectWorker()));
            }

            MasterStartContainerResponse response = (MasterStartContainerResponse) FutureUtils.awaitResult(currentSelection, request, 10);
            if (response.getSuccess()) {
                taskStatusMap.put(uniqueId, TaskStatus.INIT);
                logCollector.collect(uniqueId, ClientLoggerMessage.STANDALONE_INSTANCE_INIT_SUCCESS);
            } else {
                LOGGER.error("submitTask error.", response.getMessage());
                taskStatusMap.put(uniqueId, TaskStatus.FAILED);
                logCollector.collect(uniqueId, ClientLoggerMessage.appendMessage(ClientLoggerMessage.STANDALONE_INSTANCE_INIT_FAIL,
                        response.getMessage()));
            }

            init();
        } catch (Throwable e) {
            LOGGER.error("submitTask error.", e);
            logCollector.collect(uniqueId, ClientLoggerMessage.STANDALONE_INSTANCE_INIT_FAIL, e);
            taskStatusMap.put(uniqueId, TaskStatus.FAILED);
        }
    }

    /**
     * 轮询获取可执行机器
     * @return
     */
    protected String selectWorker() {
        List<String> allWorkers = jobInstanceInfo.getAllWorkers();
        int size = allWorkers.size();
        int index = 0;
        if (size == 0) {
            return null;
        } else if (getSerialNum() >= size) {
            index = new Long(getSerialNum() % size).intValue();
        }
        String worker = allWorkers.get(index);
        return worker;
    }

    @Override
    public synchronized void killInstance(String reason) {
        super.killInstance(reason);

        MasterKillContainerResponse response = null;
        String uniqueId = IdUtil.getUniqueIdWithoutTask(jobInstanceInfo.getJobId(), jobInstanceInfo.getJobInstanceId());
        try {
            MasterKillContainerRequest request = MasterKillContainerRequest.newBuilder()
                                                                           .setJobId(jobInstanceInfo.getJobId())
                                                                           .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                                                                           .build();
            response = (MasterKillContainerResponse)FutureUtils.awaitResult(currentSelection, request, 10);
            if (response.getSuccess()) {
                LOGGER.info("[StandaloneTaskMaster]kill standalone instance success, uniqueId:{}",uniqueId);
            } else {
                LOGGER.warn("[StandaloneTaskMaster]kill standalone instance failed, uniqueId:{}",uniqueId);
            }
        } catch (Throwable e) {
            LOGGER.error("[StandaloneTaskMaster]kill standalone instance exception uniqueId:{}",uniqueId, e);
        }

        updateNewInstanceStatus(getSerialNum(), jobInstanceInfo.getJobInstanceId(), InstanceStatus.FAILED, reason);
        if(!instanceStatus.isFinish()){
            instanceStatus = InstanceStatus.FAILED;
        }
    }

    @Override
    public synchronized void destroyContainerPool() {
        try {
            MasterDestroyContainerPoolRequest request = MasterDestroyContainerPoolRequest.newBuilder()
                .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                .setSerialNum(getSerialNum())
                .build();
            FutureUtils.awaitResult(currentSelection, request, 5);
        } catch (Throwable e) {
            LOGGER.error("destroy containerPool failed", e);
        }
    }

    @Override
    protected void checkProcessor() throws Exception {
        if ("java".equalsIgnoreCase(jobInstanceInfo.getJobType())) {
            JobProcessor processor = JavaProcessorProfileUtil.getJavaProcessor(jobInstanceInfo.getContent());
            if (processor instanceof MapJobProcessor) {
                throw new IOException(processor.getClass().getName() + " shouldn't extends MapJobProcessor or MapReduceJobProcessor");
            }
        }
    }

    public ActorSelection getCurrentSelection() {
        return currentSelection;
    }
}
