package com.alibaba.schedulerx.worker.processor;

import akka.actor.ActorSelection;
import akka.actor.Address;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.common.util.StringUtils;
import com.alibaba.schedulerx.protocol.Worker;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.batch.ContainerStatusReqHandlerPool;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.domain.DataworksProcessorProfile;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.util.WorkerIdGenerator;
import com.aliyun.dataworks_public20200518.models.GetDagRequest;
import com.aliyun.dataworks_public20200518.models.GetDagResponse;
import com.aliyun.dataworks_public20200518.models.GetInstanceLogRequest;
import com.aliyun.dataworks_public20200518.models.GetInstanceLogResponse;
import com.aliyun.dataworks_public20200518.models.GetInstanceRequest;
import com.aliyun.dataworks_public20200518.models.GetInstanceResponse;
import com.aliyun.dataworks_public20200518.models.ListManualDagInstancesRequest;
import com.aliyun.dataworks_public20200518.models.ListManualDagInstancesResponse;
import com.aliyun.dataworks_public20200518.models.ListManualDagInstancesResponseBody;
import com.aliyun.dataworks_public20200518.models.RestartInstanceRequest;
import com.aliyun.dataworks_public20200518.models.RestartInstanceResponse;
import com.aliyun.dataworks_public20200518.models.RunManualDagNodesRequest;
import com.aliyun.dataworks_public20200518.models.RunManualDagNodesResponse;
import com.aliyun.dataworks_public20200518.models.StopInstanceRequest;
import com.aliyun.teaopenapi.models.Config;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.configuration.Configuration;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * dataworks任务
 * @author yaohui
 * @create 2021/11/29 下午5:48
 **/
public class DataWorksProcessor extends JavaProcessor {

    enum DataWorksStatus {
        FAILURE, SUCCESS
    }

    private static final List<String> COMMON_REGIONS = Lists.newArrayList("cn-qingdao", "cn-huhehaote");

    private static final Long CHECK_TIME = 30L;

    protected static final Logger logger = LogFactory.getLogger(DataWorksProcessor.class);

    private LogCollector logCollector = LogCollectorFactory.get();

    private com.aliyun.dataworks_public20200518.Client createClient(JobContext context) throws Exception {
        Configuration conf = ConfigUtil.getWorkerConfig();
        String accessKey = conf.getString(WorkerConstants.ALIYUN_ACESSKEY);
        if(StringUtils.isEmpty(accessKey)){
            throw new RuntimeException("execute dataworks task ak can't be null.");
        }
        String secretKey = conf.getString(WorkerConstants.ALIYUN_SECRETKEY);
        if(StringUtils.isEmpty(secretKey)){
            throw new RuntimeException("execute dataworks task sk can't be null.");
        }
        DataworksProcessorProfile profile = JsonUtil.fromJson(context.getContent(), DataworksProcessorProfile.class);
        Config config = new Config().setAccessKeyId(accessKey).setAccessKeySecret(secretKey);
        if(COMMON_REGIONS.contains(profile.getRegion())){
            config.endpoint = "dataworks.aliyuncs.com";
        }else {
            config.endpoint = "dataworks." + profile.getRegion() + ".aliyuncs.com";
        }
        com.aliyun.dataworks_public20200518.Client client = new com.aliyun.dataworks_public20200518.Client(config);
        return client;
    }

    @Override
    public ProcessResult process(JobContext context) throws Exception {

        com.aliyun.dataworks_public20200518.Client client = createClient(context);
        DataworksProcessorProfile profile = JsonUtil.fromJson(context.getContent(), DataworksProcessorProfile.class);

        Long dagId = null;
        Long nodeInstanceId = null;
        JSONObject paramsJson = null;

        try {
            String jobInstanceParameters = context.getInstanceParameters();
            paramsJson = JSON.parseObject(jobInstanceParameters);
        }catch (Exception e){
            logger.warn("dataworks job instance parameters parse warning.");
        }

        if(paramsJson != null && paramsJson.getLong("nodeInstanceId") != null){
            // 节点实例id不为空则说明为重跑
            dagId = paramsJson.getLong("dataWorksDagId");
            nodeInstanceId = paramsJson.getLong("nodeInstanceId");
            // 获取实例信息
            GetInstanceRequest getInstanceRequest = new GetInstanceRequest()
                    .setInstanceId(nodeInstanceId)
                    .setProjectEnv(profile.getProjectEnv());
            GetInstanceResponse getInstanceResponse = client.getInstance(getInstanceRequest);
            if(getInstanceResponse.getBody().getSuccess()) {
                String status = getInstanceResponse.getBody().getData().getStatus();
                if(DataWorksStatus.FAILURE.name().equals(status) || DataWorksStatus.SUCCESS.name().equals(status)) {
                    // 实例已完成则进行实例重跑
                    RestartInstanceRequest restartInstanceRequest = new RestartInstanceRequest()
                            .setInstanceId(nodeInstanceId)
                            .setProjectEnv(profile.getProjectEnv());
                    RestartInstanceResponse restartInstanceResponse = client.restartInstance(restartInstanceRequest);
                    if (!restartInstanceResponse.getBody().getData()) {
                        DataWorksResultInfo resultInfo = new DataWorksResultInfo(dagId,nodeInstanceId, "Execute dataworks task failed, instance id return null.");
                        return new ProcessResult(false, resultInfo.toString());
                    }
                }
            }else {
                DataWorksResultInfo resultInfo = new DataWorksResultInfo(dagId,nodeInstanceId,"Execute dataworks task restart failed. "+getInstanceResponse.getBody().getErrorMessage());
                return new ProcessResult(false, resultInfo.toString());
            }
        }else {
            RunManualDagNodesRequest runManualDagNodesRequest = new RunManualDagNodesRequest()
                    .setProjectEnv(profile.getProjectEnv())
                    .setProjectName(profile.getProjectName())
                    .setBizDate(context.getDataTime().minusDays(1).toString(CommonConstants.DATE_TIME_PATTERN))
                    .setIncludeNodeIds(profile.getTaskIds())
                    .setNodeParameters(context.getJobParameters())
                    .setFlowName(profile.getFlowName());

            // 复制代码运行请自行打印 API 的返回值
            RunManualDagNodesResponse response = client.runManualDagNodes(runManualDagNodesRequest);
            if (response.getBody() != null) {
                dagId = response.getBody().getDagId();
                if (dagId != null && dagId > 0) {
                    JSONObject jsonObject = new JSONObject();
                    jsonObject.put("dataWorksDagId", dagId);

                    ListManualDagInstancesRequest listManualDagInstancesRequest = new ListManualDagInstancesRequest()
                            .setProjectEnv(profile.getProjectEnv())
                            .setProjectName(profile.getProjectName())
                            .setDagId(dagId.toString());
                    ListManualDagInstancesResponse dagInstancesResponse = client.listManualDagInstances(listManualDagInstancesRequest);
                    List<ListManualDagInstancesResponseBody.ListManualDagInstancesResponseBodyInstances> list = dagInstancesResponse.getBody().getInstances();
                    if (CollectionUtils.isNotEmpty(list)) {
                        nodeInstanceId = list.get(0).getInstanceId();
                        jsonObject.put("nodeInstanceId", nodeInstanceId);
                    } else {
                        DataWorksResultInfo resultInfo = new DataWorksResultInfo(dagId,nodeInstanceId,"Execute dataworks task failed, node instance not exists.");
                        return new ProcessResult(false, resultInfo.toString());
                    }
                    DataWorksResultInfo resultInfo = new DataWorksResultInfo(dagId, nodeInstanceId,null);
                    reportTaskStatus(context, new ProcessResult(InstanceStatus.RUNNING, resultInfo.toString()), resultInfo.toString());
                } else {
                    DataWorksResultInfo resultInfo = new DataWorksResultInfo(dagId, nodeInstanceId,"Execute dataworks task failed, instance id return null.");
                    return new ProcessResult(false, resultInfo.toString());
                }
            }
        }

        while (true) {
            TimeUnit.SECONDS.sleep(CHECK_TIME);
            GetDagRequest getDagRequest = new GetDagRequest()
                    .setDagId(dagId).setProjectEnv(profile.getProjectEnv());
            // 复制代码运行请自行打印 API 的返回值
            GetDagResponse res = client.getDag(getDagRequest);
            String status = res.getBody().getData().getStatus();
            if (DataWorksStatus.FAILURE.name().equals(status)) {
                GetInstanceLogRequest getInstanceLogRequest = new GetInstanceLogRequest()
                        .setInstanceId(nodeInstanceId).setProjectEnv(profile.getProjectEnv());
                // 复制代码运行请自行打印 API 的返回值
                GetInstanceLogResponse logResponse = client.getInstanceLog(getInstanceLogRequest);
                String message = null;
                if(logResponse != null && logResponse.getBody().success && StringUtils.isNotEmpty(logResponse.getBody().getData())){
                    String[] array = logResponse.getBody().getData().split("\n");
                    int start = Math.max(0, array.length-8);
                    message = array[start];
                }
                logCollector.collect(context.getUniqueId(), logResponse.getBody().getData());
                DataWorksResultInfo resultInfo = new DataWorksResultInfo(dagId, nodeInstanceId, "Execute dataworks task failed. case by:"+message);
                return new ProcessResult(false, resultInfo.toString());
            } else if (DataWorksStatus.SUCCESS.name().equals(status)) {
                GetInstanceLogRequest getInstanceLogRequest = new GetInstanceLogRequest()
                        .setInstanceId(nodeInstanceId)
                        .setProjectEnv(profile.getProjectEnv());
                // 复制代码运行请自行打印 API 的返回值
                GetInstanceLogResponse logResponse = client.getInstanceLog(getInstanceLogRequest);
                logCollector.collect(context.getUniqueId(), logResponse.getBody().getData());
                DataWorksResultInfo resultInfo = new DataWorksResultInfo(dagId, nodeInstanceId, "Execute dataworks task success.");
                return new ProcessResult(true, resultInfo.toString());
            }
        }
    }


    @Override
    public void kill(JobContext context) {
        try {
            DataworksProcessorProfile profile = JsonUtil.fromJson(context.getContent(), DataworksProcessorProfile.class);
            String jobInstanceParameters = context.getInstanceParameters();
            JSONObject paramsJson = JSON.parseObject(jobInstanceParameters);
            Long nodeInstanceId = paramsJson.getLong("nodeInstanceId");
            StopInstanceRequest stopInstanceRequest = new StopInstanceRequest()
                    .setInstanceId(nodeInstanceId)
                    .setProjectEnv(profile.getProjectEnv());
            com.aliyun.dataworks_public20200518.Client client = createClient(context);
            client.stopInstance(stopInstanceRequest);
        }catch (Exception e){
            logger.error("dataworks job instance stop failed.");
        }
    }


    /**
     * 汇报处理信息
     * @param context
     * @param result
     */
    private void reportTaskStatus(JobContext context, ProcessResult result, String progress) {
        Worker.ContainerReportTaskStatusRequest.Builder resultBuilder = Worker.ContainerReportTaskStatusRequest.newBuilder();
        resultBuilder.setJobId(context.getJobId());
        resultBuilder.setJobInstanceId(context.getJobInstanceId());
        resultBuilder.setTaskId(context.getTaskId());
        resultBuilder.setStatus(result.getStatus().getValue());
        Address address = SchedulerxWorker.actorSystem.provider().getDefaultAddress();
        String workerAddr = address.host().get() + ":" + address.port().get();
        resultBuilder.setWorkerAddr(workerAddr);
        resultBuilder.setWorkerId(WorkerIdGenerator.get());
        resultBuilder.setSerialNum(context.getSerialNum());
        resultBuilder.setInstanceMasterActorPath(context.getInstanceMasterActorPath());
        if (context.getTaskName() != null) {
            resultBuilder.setTaskName(context.getTaskName());
        }
        if (result.getResult() != null) {
            resultBuilder.setResult(result.getResult());
        }
        resultBuilder.setProgress(progress);

        boolean enableShareContainerPool = ConfigUtil.getWorkerConfig().getBoolean(WorkerConstants.SHARE_CONTAINER_POOL, false);
        boolean submitResult = false;
        if (enableShareContainerPool) {
            submitResult = ContainerStatusReqHandlerPool.INSTANCE.submitReq(0, resultBuilder.build());
        } else {
            submitResult = ContainerStatusReqHandlerPool.INSTANCE.submitReq(context.getJobInstanceId(), resultBuilder.build());
        }
        logger.info("reportTaskStatus instanceId={} submitResult={}, processResult={}", context.getUniqueId(),
                submitResult, result);
        if (!submitResult) {
            ActorSelection masterActorSelection = SchedulerxWorker.actorSystem.actorSelection(context.getInstanceMasterActorPath());
            masterActorSelection.tell(resultBuilder.build(), null);
        }
    }


    /**
     * Dataworks执行结果信息
     */
    class DataWorksResultInfo {

        private Long dataWorksDagId;

        private Long nodeInstanceId;

        private String url;

        private String message;

        public DataWorksResultInfo(Long dataWorksDagId, Long nodeInstanceId, String message) {
            this.dataWorksDagId = dataWorksDagId;
            this.nodeInstanceId = nodeInstanceId;
            this.message = message;
        }

        public Long getDataWorksDagId() {
            return dataWorksDagId;
        }

        public void setDataWorksDagId(Long dataWorksDagId) {
            this.dataWorksDagId = dataWorksDagId;
        }

        public Long getNodeInstanceId() {
            return nodeInstanceId;
        }

        public void setNodeInstanceId(Long nodeInstanceId) {
            this.nodeInstanceId = nodeInstanceId;
        }

        public String getUrl() {
            return url;
        }

        public void setUrl(String url) {
            this.url = url;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }

        @Override
        public String toString() {
            return JSON.toJSONString(this);
        }
    }
}
